/* 	THexEditor.java
 *	Copyright (c) 2009, Brains2B.org
 *	
 *	Created by: Dennis Groenendijk
 *	Created on: Oct 19, 2009
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that  * the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions and 
 * the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
 * the following disclaimer in the documentation and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products derived from this software 
 * without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY 
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.brains2b.thex.editor;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ContainerEvent;
import java.awt.event.ContainerListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.ResourceBundle;

import javax.swing.ActionMap;
import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.plaf.ColorUIResource;
import javax.swing.plaf.FontUIResource;

import org.brains2b.swingext.action.ExtActionMap;
import org.brains2b.swingext.dialog.AboutPane;
import org.brains2b.swingext.dialog.DialogHelper;
import org.brains2b.swingext.gui.GuiItemFactory;
import org.brains2b.swingext.preference.PreferencePane;
import org.brains2b.thex.BinaryDocument;
import org.brains2b.thex.HexEditor;
import org.brains2b.thex.HexEditorMap;
import org.brains2b.thex.io.BinaryIOKit;
import org.brains2b.util.GuiHelper;
import org.brains2b.util.StringHelper;

/**
 * Wrapper Application around Thex making it a MDI
 * <UL>THexEditor handles
 * <LI>Opening multiple instances of {@link HexEditor} in tab form
 * <LI>Controls menu 
 * <LI>Allows for files to be passed through the commandline
 *</UL>
 * @author <A HREF="MAILTO:dennis@brains2b.nl">dennis@brains2b.nl</A>
 * @version $Revision$ $Date$
 */
public class THexEditor extends JFrame implements ActionListener {
	
	/**
	 * Handler for window closing
	 * @see THexEditor#close();
	 */
	class WindowHandler extends WindowAdapter {
		
		public void windowClosing(WindowEvent e) {
			close();
		}
	}
	
	/**
	 * MouseHandler to control the Popup menu for the tabs
	 */
	class MouseHandler extends MouseAdapter {
		
		@Override
		public void mousePressed(MouseEvent e) {
			if (e.isPopupTrigger()) {
				final int idx = m_tab.indexAtLocation(e.getPoint().x,e.getPoint().y);
				if (idx==-1) return;
				m_tab.setSelectedIndex(idx);
				m_tabPopup.show(m_tab, e.getPoint().x, e.getPoint().y);
			}
		}
		
	}
	
	/**
	 * TabHandler controls the Tab menu showing the different open documents
	 */
	class TabHandler implements ActionListener, ContainerListener {
		public void actionPerformed(ActionEvent e) {
			String ac = e.getActionCommand();
			int idx = Integer.parseInt(ac.substring(7));
			m_tab.setSelectedIndex(idx);
		}

		public void componentAdded(ContainerEvent e) {
			HexEditor h = (HexEditor) e.getChild();
			int idx = m_tab.indexOfComponent(h);
			JMenuItem m = new JMenuItem(h.toString());
			m.setActionCommand("Window." +idx);
			m.addActionListener(this);
			m_tabMenu.insert(m,idx);	
		}

		public void componentRemoved(ContainerEvent e) {
			for (int i=0;i<m_tab.getTabCount();i++) {
				m_tabMenu.remove(i);
				HexEditor h = (HexEditor) m_tab.getComponentAt(i);
				JMenuItem m = new JMenuItem(h.toString());
				m.setActionCommand("Window." +i);
				m_tabMenu.insert(m,i);
			}
			m_tabMenu.remove(m_tab.getTabCount());
		}
	}
	
	/**
	 * ActionHandler handling the different Tab action commands
	 */
	class TabActionHandler implements ActionListener {

		public void actionPerformed(ActionEvent e) {
			final String ac = e.getActionCommand();
			if ("Tab.Close".equals(ac)) {
				HexEditor editor = getCurrentEditor();
				if (editor.isChanged()) {
					int result = JOptionPane.showConfirmDialog(m_tab, m_resource.getString("SaveAs.Confirm.Message"),m_resource.getString("SaveAs.Confirm.Title"),JOptionPane.YES_NO_CANCEL_OPTION);
					switch (result) {
					case JOptionPane.YES_OPTION:
						saveFile(editor.getIoKit(), editor.getIoKit().getFile(), editor.getDocument());
					case JOptionPane.NO_OPTION:
						m_tab.removeTabAt(m_tab.indexOfComponent(editor));
					default:
					}
				} else {
					m_tab.removeTabAt(m_tab.indexOfComponent(editor));
				}
				if (m_tab.getTabCount()==0) {
					createNewFile();
				}
			} else if ("Tab.CloseAll".equals(ac)) {
				Object[] editors = getSaveChangedEditors();
        		if (editors==null) return; //close all is canceled
        		for (int i=0;i<editors.length;i++) {
        			HexEditor editor = (HexEditor) editors[i];
        			saveFile(editor.getIoKit(),editor.getIoKit().getFile(),editor.getDocument());
        		}
        		for (int i=m_tab.getTabCount()-1;i>=0;i--) {
        			m_tab.removeTabAt(i);
        		}
        		createNewFile();
			} else if ("Tab.CloseExcept".equals(ac)) {
				HexEditor current = getCurrentEditor();
				int currentIdx = m_tab.indexOfComponent(current);
				Object[] editors = getSaveChangedEditors();
        		if (editors==null) return; //close all is canceled
        		for (int i=0;i<editors.length;i++) {
        			if (editors[i]!=current) {
        				HexEditor editor = (HexEditor) editors[i];
        				saveFile(editor.getIoKit(),editor.getIoKit().getFile(),editor.getDocument());
        			}
        		}
        		for (int i=m_tab.getTabCount()-1;i>=0;i--) {
        			if (i!=currentIdx) {
        				m_tab.removeTabAt(i);
        			}
        		}
			}
			
		}
		
	}
	
	/**
	 * ActionHandler to react to opening Recent files
	 */
	class RecentHandler implements ActionListener {
	
		public void actionPerformed(ActionEvent e) {
			JMenuItem itm = (JMenuItem) e.getSource();
			File f = new File(itm.getText());
			if (!f.exists()) {
				JOptionPane.showMessageDialog(THexEditor.this, m_resource.getString("OpenFile.Message").replaceFirst("%1", itm.getText()));
			}
			openFile(f);
			
		}
				
	}
	
	final ResourceBundle m_resource = ResourceBundle.getBundle("org.brains2b.thex.resource.thexeditor");
	private boolean m_showAscii;
	private ExtActionMap m_map;
	private JTabbedPane m_tab;
	private JPopupMenu m_tabPopup;
	private JMenu m_tabMenu;
	private JMenu m_recentMenu;
	private RecentHandler m_recentHandler;
	private THexPreference m_prefs;
	
	/**
	 * Constructor
	 */
	public THexEditor() {
		super();
		init();
	}
	
	/**
	 * Application entry point
	 * @param args String[] allows for multiple file names and the following options
	 * <li>-n or --noascii<tab>To hide the ascii view when opening the THexEditor</li>
	 * 
	 */
	public static void main(String[] args) {
		THexEditor editor = new THexEditor();
		editor.setVisible(true);
		editor.handleArgs(args);
	}
	
	private void init() {
		m_prefs = new THexPreference();
		setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
		addWindowListener(new WindowHandler());
		setLayout(new BorderLayout());
		setTitle("THex Editor");
		createMap();
		m_tab = new JTabbedPane();
		m_tab.setActionMap(createTabMap());
		m_tab.addMouseListener(new MouseHandler());
		m_tab.addContainerListener(new TabHandler());
		add(m_tab,BorderLayout.CENTER);
		m_tabPopup = new JPopupMenu();
		GuiItemFactory.createMenuItem(m_tabPopup, m_tab.getActionMap().get("Tab.Close"));
		GuiItemFactory.createMenuItem(m_tabPopup, m_tab.getActionMap().get("Tab.CloseExcept"));
		GuiItemFactory.createMenuItem(m_tabPopup, m_tab.getActionMap().get("Tab.CloseAll"));
		createMenu();
		createNewFile();
		loadDefaults();
	}
	
	private void handleArgs(String[] args) {
		if (args==null || args.length==0) return;
		String lastAct ="open";
		for (int i=0;i<args.length;i++) {
			if (args[i].startsWith("-")) {
				if (args[i].charAt(1)=='n') {
					m_showAscii = false;
				}
			} else if (args[i].startsWith("--")) {
				if ("noascii".equals(args[i].substring(2))) {
					m_showAscii = false;
				}
			} else if ("open".equals(lastAct)) {
        		File f = new File(args[i]);
        		if (f.exists()) {
    				openFile(f);
        		} else {
        			JOptionPane.showMessageDialog(this, m_resource.getString("OpenFile.Message").replaceFirst("%1", args[i]));
        		}
			}
		}
	}
	
	private void createMap() {
		m_map = new ExtActionMap(this);	
		m_map.addAction("File.New",m_resource.getString("File.New.Label"), m_resource.getString("File.New.Tooltip"),  m_resource.getString("File.New.Icon"), "ctrl N", 'n');
		m_map.addAction("File.Open",m_resource.getString("File.Open.Label"), m_resource.getString("File.Open.Tooltip"),  m_resource.getString("File.Open.Icon"), "ctrl O", 'o');
		m_map.addAction("File.Save",m_resource.getString("File.Save.Label"), m_resource.getString("File.Save.Tooltip"),  m_resource.getString("File.Save.Icon"), "ctrl S", 's');
		m_map.addAction("File.SaveAs",m_resource.getString("File.SaveAs.Label"), m_resource.getString("File.SaveAs.Tooltip"),  m_resource.getString("File.SaveAs.Icon"), "ctrl shift S", 'a');
		m_map.addAction("System.Exit",m_resource.getString("System.Exit.Label"), m_resource.getString("System.Exit.Tooltip"),  m_resource.getString("System.Exit.Icon"), "alt F4", 'x');
		m_map.addAction("System.Help",m_resource.getString("System.Help.Label"), m_resource.getString("System.Help.Tooltip"),  m_resource.getString("System.Help.Icon"), "F1", 'x');
		m_map.addAction("System.About",m_resource.getString("System.About.Label"), m_resource.getString("System.About.Tooltip"),  m_resource.getString("System.About.Icon"), "shift F1", 'x');
		m_map.addAction("System.Options",m_resource.getString("System.Options.Label"), m_resource.getString("System.Options.Tooltip"),  m_resource.getString("System.Options.Icon"), null, 'p');
	}
	
	private ActionMap createTabMap() {
		ExtActionMap map = new ExtActionMap(new TabActionHandler());	
		map.addAction("Tab.Close",m_resource.getString("Tab.Close.Label"), m_resource.getString("Tab.Close.Tooltip"),  m_resource.getString("Tab.Close.Icon"), "ctrl W", '0');		
		map.addAction("Tab.CloseAll",m_resource.getString("Tab.CloseAll.Label"), m_resource.getString("Tab.CloseAll.Tooltip"),  m_resource.getString("Tab.CloseAll.Icon"), "ctrl shift W", '0');
		map.addAction("Tab.CloseExcept",m_resource.getString("Tab.CloseExcept.Label"), m_resource.getString("Tab.CloseExcept.Tooltip"),  m_resource.getString("Tab.CloseExcept.Icon"), "ctrl shift F6", '0');
		map.addAction("Tab.CloseSelected",m_resource.getString("Tab.CloseSelected.Label"), m_resource.getString("Tab.CloseSelected.Tooltip"),  m_resource.getString("Tab.CloseSelected.Icon"), "ctrl F6", '0');
		return map;
	}

	private void createMenu() {
		ResourceBundle rs = ResourceBundle.getBundle("org.brains2b.thex.resource.thexeditor");
		JMenuBar mb = new JMenuBar();
		mb.setActionMap(m_map);
		JMenu fl = GuiItemFactory.createMenu(rs.getString("File.Menu"), 'f');
		GuiItemFactory.createMenuItem(fl, m_map.get("File.New"));
		GuiItemFactory.createMenuItem(fl, m_map.get("File.Open"));
		m_recentMenu = new JMenu(m_resource.getString("File.OpenRecent.Label"));
		m_recentMenu.setToolTipText(m_resource.getString("File.OpenRecent.Tooltip"));
		m_recentMenu.setMnemonic('r');
		m_recentHandler = new RecentHandler();
		fl.add(m_recentMenu);
		if (!StringHelper.isEmpty(m_prefs.getProperty("recent.files"))) {
			addRecentFile(m_prefs.getProperty("recent.files").split(";"));
		}
		GuiItemFactory.createMenuItem(fl, m_map.get("File.Save"));
		GuiItemFactory.createMenuItem(fl, m_map.get("File.SaveAs"));
		fl.addSeparator();
		GuiItemFactory.createMenuItem(fl, m_map.get("System.Options"));
		fl.addSeparator();
		GuiItemFactory.createMenuItem(fl, m_map.get("System.Exit"));
		mb.add(fl);
		
		
		HexEditorMap map = new HexEditorMap(this);
		JMenu edt = GuiItemFactory.createMenu(rs.getString("Edit.Menu"), 'e');
		GuiItemFactory.createMenuItem(edt, map.get("Edit.Undo"));
		GuiItemFactory.createMenuItem(edt, map.get("Edit.Redo"));
		edt.addSeparator();
		GuiItemFactory.createMenuItem(edt, map.get("Edit.Cut"));
		GuiItemFactory.createMenuItem(edt, map.get("Edit.Copy"));
		GuiItemFactory.createMenuItem(edt, map.get("Edit.Paste"));
		edt.addSeparator();
		GuiItemFactory.createMenuItem(edt, map.get("Edit.SelectAll"));
		edt.addSeparator();
		GuiItemFactory.createMenuItem(edt, map.get("Edit.Find"));
		GuiItemFactory.createMenuItem(edt, map.get("Edit.FindNext"));
		GuiItemFactory.createMenuItem(edt, map.get("Edit.FindPrev"));
		GuiItemFactory.createMenuItem(edt, map.get("Edit.Goto"));
		mb.add(edt);
		
		/*MarkMap mmap = new MarkMap(this);
		JMenu mark = GuiItemFactory.createMenu(rs.getString("Mark.Menu"), 'm');
		GuiItemFactory.createMenuItem(mark, mmap.get("Mark.Add"));
		GuiItemFactory.createMenuItem(mark, mmap.get("Mark.Edit"));
		GuiItemFactory.createMenuItem(mark, mmap.get("Mark.Remove"));
		mb.add(mark);*/
		
		m_tabMenu = GuiItemFactory.createMenu(rs.getString("Window.Menu"), 'w');
		m_tabMenu.addSeparator();
		GuiItemFactory.createMenuItem(m_tabMenu, m_tab.getActionMap().get("Tab.Close"));
		GuiItemFactory.createMenuItem(m_tabMenu, m_tab.getActionMap().get("Tab.CloseExcept"));
		GuiItemFactory.createMenuItem(m_tabMenu, m_tab.getActionMap().get("Tab.CloseSelected"));
		GuiItemFactory.createMenuItem(m_tabMenu, m_tab.getActionMap().get("Tab.CloseAll"));
		mb.add(m_tabMenu);
		
		JMenu hl = GuiItemFactory.createMenu(rs.getString("Help.Menu"), 'h');
		GuiItemFactory.createMenuItem(hl, m_map.get("System.Help"));
		GuiItemFactory.createMenuItem(hl, m_map.get("System.About"));	
		mb.add(hl);
		setJMenuBar(mb);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void actionPerformed(ActionEvent e) {
		final String ac = e.getActionCommand();
		if (ac.startsWith("Edit")) {
			getCurrentEditor().actionPerformed(e);
		} else if ("System.Exit".equals(ac)) {
			close();			
		} else if ("System.Help".equals(ac)) {
			//TODO Implement help	
		} else if ("System.Options".equals(ac)) {
			saveDefaults();
			PreferencePane pane = new PreferencePane(m_prefs);
			pane.registerResourceBundle(m_resource);
			pane.addIgnorePattern("size");
			pane.addIgnorePattern("recent");
			pane.setPropertyType("thex.color.select", PreferencePane.COLOR);
			pane.setPropertyType("thex.color.foreground", PreferencePane.COLOR);
			pane.setPropertyType("thex.color.background", PreferencePane.COLOR);
			DialogHelper.showDialog(this, pane, m_resource.getString("PreferencePane.Title"));
			if (pane.isChanged()) {
				loadDefaults();
			}
		} else if ("System.About".equals(ac)) {
			AboutPane.showDialog(this);
		} else if ("File.New".equals(ac)) {
			createNewFile();
		} else if ("File.Open".equals(ac)) {
			openFile();
		} else if ("File.Save".equals(ac)) {
			HexEditor editor = getCurrentEditor();
			saveFile(editor.getIoKit(),editor.getIoKit().getFile(),editor.getDocument());
			editor.setTitle(null);
		} else if ("File.SaveAs".equals(ac)) {
			HexEditor editor = getCurrentEditor();
			saveFile(editor.getIoKit(),null,editor.getDocument());
			int idx = m_tab.indexOfComponent(editor);
			m_tab.setTitleAt(idx, editor.getIoKit().getFile().getName());
		} else {
			/*MarkManager mm = getCurrentEditor().getPluginForCommand(ac);
			if (mm!=null) {
				mm.actionPerformed(e);
			}*/
		}
	}
	
	private void close() {
		Object[] editors = getSaveChangedEditors();
		if (editors==null) return; //close is canceled
		for (int i=0;i<editors.length;i++) {
			HexEditor editor = (HexEditor) editors[i];
			saveFile(editor.getIoKit(),editor.getIoKit().getFile(),editor.getDocument());
		}
		saveDefaults();
		dispose();
	}
	
	private Object[] getSaveChangedEditors() {
		final ArrayList list = new ArrayList();
		for (int i=0;i<m_tab.getTabCount();i++) {
			HexEditor he = (HexEditor) m_tab.getComponentAt(i);
			if (he.isChanged()) {
				list.add(m_tab.getComponentAt(i));
			}
		}
		
		if (list.isEmpty()) return new HexEditor[0];
		
		final JPanel save = new JPanel();
		save.setLayout(new BorderLayout(10,10));
		final JList sl = new JList(list.toArray());
		final BitSet result = new BitSet(1);
		save.add(new JLabel(m_resource.getString("SaveForm.Label")),BorderLayout.NORTH);
		sl.setMinimumSize(new Dimension(200,100));
		sl.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
		save.add(new JScrollPane(sl),BorderLayout.CENTER);
		JPanel buttons = new JPanel();
		buttons.setLayout(new BorderLayout(10,5));
		JButton ok = new JButton(m_resource.getString("SaveForm.Ok"));
		ok.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				DialogHelper.getWindowPane(save).dispose();	
			}
		});
		buttons.add(ok,BorderLayout.WEST);
		JButton cancel = new JButton(m_resource.getString("SaveForm.Cancel"));
		cancel.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				result.set(0);
				sl.clearSelection();
				DialogHelper.getWindowPane(save).dispose();	
			}
		});
		buttons.add(cancel,BorderLayout.EAST);
		JButton invert = new JButton(m_resource.getString("SaveForm.Invert"));
		invert.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				int[] sel = sl.getSelectedIndices();
				int idx=0;
				if (sel.length ==0) {
					sl.addSelectionInterval(0, list.size()-1);
					return;
				}
				for (int i=0;i<list.size();i++) {
					if (i==sel[idx]) {
						sl.removeSelectionInterval(sel[idx], sel[idx]);
					} else {
						sl.addSelectionInterval(i, i);
					}
				}
					
			}
		});
		buttons.add(invert,BorderLayout.CENTER);
		
		save.add(buttons,BorderLayout.SOUTH);
		
		DialogHelper.showDialog(this, save, m_resource.getString("SaveForm.Title"));
		if (result.get(0)) return null;
		return sl.getSelectedValues();
		
	}
	
	private void createNewFile() {
		HexEditor editor= new HexEditor();
		//editor.registerPlugin(new MarkManager());
		editor.setTitle(m_resource.getString("File.Empty"));
		editor.setAsciiVisible(m_showAscii);
		m_tab.add(editor.getTitle(), editor);
		m_tab.setSelectedComponent(editor);
	}
	
	
	private void openFile() {
		JFileChooser jfc = new JFileChooser();
		jfc.setMultiSelectionEnabled(true);
		int result = jfc.showOpenDialog(this);
		if (result==JFileChooser.CANCEL_OPTION) return;
		
		File[] f= jfc.getSelectedFiles();
		for (int i=0;i<f.length;i++) {
			openFile(f[i]);
		}
		
	}
	
	private void openFile(File f) {
		HexEditor editor = new HexEditor();
		editor.getActionMap().setParent(m_map);
		try {
			editor.getIoKit().open(f, editor.getDocument());
		} catch (Exception ex) {
			
		}
		editor.setAsciiVisible(m_showAscii);
		//editor.registerPlugin(new MarkManager());
		m_tab.add(f.getName(), editor);
		m_tab.setSelectedIndex(m_tab.getTabCount()-1);
		addRecentFile(f.getAbsolutePath());
	}
	
	/**
	 * save the {@link BinaryDocument} to the given file.
	 * <p>While the document is a representation, the IOKit is also supplied to read the actual content to write to a (new) file
	 * @param kit BinaryIOKit
	 * @param f File
	 * @param doc BinaryDocument
	 */
	protected void saveFile(BinaryIOKit kit, File f, BinaryDocument doc) {
		if (f==null) {
			JFileChooser jfc = new JFileChooser();
			jfc.setMultiSelectionEnabled(false);
			int result = jfc.showSaveDialog(this);
			if (result==JFileChooser.CANCEL_OPTION) return;
			f= jfc.getSelectedFile();
		}
		try {
			kit.save(f, doc);
		} catch (Exception ex) {}
	}
	
	/**
	 * get the HexEditor that is currently on the selected tab 
	 * @return HexEditor
	 */
	protected HexEditor getCurrentEditor() {
		return (HexEditor) m_tab.getSelectedComponent();
	}
	
	private void saveDefaults() {
		m_prefs.setProperty("thex.application.size", GuiHelper.encodeDimension(getSize()));
		StringBuffer recents = new StringBuffer();
		for (int i=0;i<m_recentMenu.getMenuComponentCount();i++) {
			if (recents.length()>0) {
				recents.append(";");
			}
			recents.append(((JMenuItem) m_recentMenu.getMenuComponent(i)).getText());
			
		}
		m_prefs.setProperty("recent.files", recents.toString());
		m_prefs.store();
		
	}
	private void loadDefaults() {
		Dimension d = GuiHelper.decodeDimension(m_prefs.getProperty("thex.application.size"));
		if (d!=null) {
			setSize(d);
		}
		if (!StringHelper.isEmpty(m_prefs.getProperty("thex.ascii.visible"))) {
			m_showAscii = Boolean.parseBoolean(m_prefs.getProperty("thex.ascii.visible"));
			if (getCurrentEditor()!=null) {
				getCurrentEditor().setAsciiVisible(m_showAscii);
			}
		}
		if (!StringHelper.isEmpty(m_prefs.getProperty("thex.font.binary"))) {
			UIManager.put("EditorPane.font", new FontUIResource(Font.decode(m_prefs.getProperty("thex.font.binary"))));
		}
		/*if (!StringHelper.isEmpty(m_prefs.getProperty("thex.font.ascii"))) {
			UIManager.put("AsciiEditor.font", new FontUIResource(Font.decode(m_prefs.getProperty("thex.font.ascii"))));
		}*/
		if (!StringHelper.isEmpty(m_prefs.getProperty("thex.color.select"))) {
			UIManager.put("EditorPane.selectionBackground", new ColorUIResource(Color.decode(m_prefs.getProperty("thex.color.select"))));
		} 
		if (!StringHelper.isEmpty(m_prefs.getProperty("thex.color.foreground"))) {
			UIManager.put("EditorPane.foreground", new ColorUIResource(Color.decode(m_prefs.getProperty("thex.color.foreground"))));
		}
		if (!StringHelper.isEmpty(m_prefs.getProperty("thex.color.background"))) {
			UIManager.put("EditorPane.background", new ColorUIResource(Color.decode(m_prefs.getProperty("thex.color.background"))));
		}
		if (getCurrentEditor()!=null) {
			SwingUtilities.updateComponentTreeUI(this);
		}
	}
	
	private void addRecentFile(String file) {
		addRecentFile(new String[] {file});
	}
	
	private void addRecentFile(String[] files) {
		for (int i=0;i<files.length;i++) {
			JMenuItem itm =new JMenuItem(files[i]);
			for (int j=0;j<m_recentMenu.getMenuComponentCount();j++) {
				if (files[i].equals(((JMenuItem) m_recentMenu.getMenuComponent(j)).getText())) {
					((JMenuItem) m_recentMenu.getMenuComponent(j)).removeActionListener(m_recentHandler);
					m_recentMenu.remove(j);
					break;
				}
			}
			itm.addActionListener(m_recentHandler);
			m_recentMenu.insert(itm,i);
		}
		for (int i=10;i<m_recentMenu.getMenuComponentCount();i++) {
			((JMenuItem) m_recentMenu.getMenuComponent(i)).removeActionListener(m_recentHandler);
			m_recentMenu.remove(i);
		}
	}
}
