/* 	LobCacheTest.java
 *	Copyright (c) 2008, Brains2B.org
 *	
 *	Created by: Dennis Groenendijk
 *	Created on: Mar 21, 2008
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that  * the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions and 
 * the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
 * the following disclaimer in the documentation and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products derived from this software 
 * without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY 
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package test.org.brains2b.data.sql.lob;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.sql.Blob;
import java.util.Arrays;

import junit.framework.TestCase;

import org.brains2b.data.sql.lob.LobCache;
import org.brains2b.data.sql.lob.LobHelper;

/**
 * Unit test for LobCache
 *
 * @author <A HREF="MAILTO:dennis@brains2b.nl">dennis@brains2b.nl</A>
 * @version 0.90 [21-03-2008]
 */
public class LobCacheTest extends TestCase {
	private final static String TEST = "<startdate>20-01-2002</startdate>";
	private LobCache m_cache;
	
	protected void setUp() throws Exception {
		m_cache = new LobCache();
		byte[] b= new byte[4096];
		int len = 0;
		OutputStream os = m_cache.setBinaryStream(0);
		FileInputStream fis = new FileInputStream("doc/project.xml");
		while ( (len=fis.read(b))!=-1) {
			os.write(b, 0, len);		
		}
		os.close();
		fis.close();
		super.setUp();
	}
	
	public void testGetBinaryStream() {
		try {
			compareStream(m_cache.getBinaryStream(),new FileInputStream("doc/project.xml"));
		} catch (Exception ex) {
			assertTrue(ex.getMessage(),false);
		}
	}

	public void testGetBytes() {
		try {
			FileInputStream fis = new FileInputStream("doc/project.xml");
			fis.skip(576L);
			byte[] b = new byte[1039];
			fis.read(b);
			fis.close();
			compareBytes(m_cache.getBytes(576L, 1039),b);
		} catch (Exception ex) {
			assertTrue(ex.getMessage(),false);
		}
	}

	public void testLength() {
		try {
			assertEquals(m_cache.length(), new File("doc/project.xml").length());
		} catch (Exception ex) {
			assertTrue(ex.getMessage(),false);
		}
		
	}
	
	public void testShortLength() {
		try {
			LobCache cache = new LobCache();
			cache.setBytes(0, TEST.getBytes());
			assertEquals(33,cache.length());
		} catch (Exception ex) {
			assertTrue(ex.getMessage(),false);
		}
		
	}

	public void testPositionBlobLong() {
		try {
			Blob blob =LobHelper.createBlob();
			m_cache.position(blob, 90);
			assertTrue("Method should have thrown exception",false);
		} catch (Exception ex) {
			assertTrue(true);
		}
	}

	public void testPositionByteArrayLong() {
		try {
			FileInputStream fis = new FileInputStream("doc/project.xml");
			fis.skip(935L);
			byte[] b = new byte[951];
			fis.read(b);
			fis.close();
			assertEquals(935L,m_cache.position(b, 90));
			assertEquals(-1,m_cache.position(b, 1024));
		} catch (Exception ex) {
			assertTrue(ex.getMessage(),false);
		}
	}

	public void testSetBinaryStream() {
		try {
			OutputStream os = m_cache.setBinaryStream(0L);
			byte[] b = new byte[159];
			Arrays.fill(b, (byte) 91);
			os.write(b);
			os.flush();
			assertEquals(b.length,m_cache.length());
			compareBytes(m_cache.getBytes(0, b.length), b);
		} catch (Exception ex) {
			assertTrue(ex.getMessage(),false);
		}
	}
	public void testSetBinaryStreamPos() {	
		try {
			byte[] keep = m_cache.getBytes(0, 901);
			OutputStream os = m_cache.setBinaryStream(901L);
			byte[] b = new byte[159];
			Arrays.fill(b, (byte) 91);
			os.write(b);
			os.flush();
			assertEquals(b.length+keep.length,m_cache.length());
			compareBytes(m_cache.getBytes(0, keep.length), keep);
			compareBytes(m_cache.getBytes(keep.length, b.length), b);
		} catch (Exception ex) {
			assertTrue(ex.getMessage(),false);
		}
	}

	public void testSetBytesLongByteArrayIntInt() {
		try {
			byte[] b = new byte[159];
			Arrays.fill(b, (byte) 91);
			long len = m_cache.length();
			m_cache.setBytes(m_cache.length(),b,10,20);
			assertEquals(len+20, m_cache.length());
			
		} catch (Exception ex) {
			assertTrue(ex.getMessage(),false);
		}
	}

	public void testSetBytesLongByteArray() {
		try {
			byte[] b = new byte[159];
			Arrays.fill(b, (byte) 91);
			long len = m_cache.length();
			m_cache.setBytes(m_cache.length(),b);
			assertEquals(len+b.length, m_cache.length());
			compareBytes(b, m_cache.getBytes(len, b.length));
		} catch (Exception ex) {
			assertTrue(ex.getMessage(),false);
		}
	}

	public void testTruncate() {
		try {
			m_cache.truncate(901);
			assertEquals(m_cache.length(), 901);
		} catch (Exception ex) {
			assertTrue(ex.getMessage(),false);
		}
	}

	private void compareStream(InputStream isA, InputStream isB) {
		byte[] bA= new byte[4096];
		byte[] bB= new byte[4096];
		int lenA = 0;
		int lenB = 0;
		try {
			while ( (lenA=isA.read(bA))!=-1) {
				lenB = isB.read(bB);
				assertEquals(lenA, lenB);
				compareBytes(bA,bB);
			}
			isA.close();
			isB.close();
		} catch (IOException iex) {
			assertTrue(iex.getMessage(),false);
		}
	}
	
	private void compareBytes(byte[] bA, byte[] bB) {
		for (int i=0;i<bA.length;i++) {
			assertEquals("Offset : " + i + " is different " + bA[i] +" " + bB[i],  bA[i],bB[i]);
		}
	}
}
