/* CursorTest.java
 * 
 * Created on: May 5, 2003
 * Created by: Dennis Groenendijk
 * 
 * Copyright (c) 2003-2004, Brains2B.org
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that
 * the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions and 
 * the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
 * the following disclaimer in the documentation and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products derived from this software 
 * without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY 
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/* @#History
* [When]		[Who]				[What]
* Jan 6, 2004	dennis@brains2b.nl	Added group test, removed references to deprecated methods
* Apr 7, 2004		dennis@brains2b.nl	rewrote test to use DEPT-EMP tables	
*/
package test.org.brains2b.data.sql.cursor;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.Calendar;
import java.util.List;

import junit.framework.TestCase;

import org.brains2b.data.DataException;
import org.brains2b.data.cursor.Cursor;
import org.brains2b.data.cursor.CursorData;
import org.brains2b.data.sql.cursor.SqlCursor;
import org.brains2b.log.Logger;
import org.brains2b.sql.LocalConnector;

/** Unit test for org.brains2b.data.cursor.Cursor
 *
 * @author <A HREF="MAILTO:dennis@brains2b.nl">dennis@brains2b.nl</A>
 * @version 0.30 [07-04-2004]
 * @see org.brains2b.data.cursor.Cursor
 */
public class SqlCursorTest extends TestCase {

	private Connection m_connection;

	/**
	 * Constructor for CursorTest.
	 * @param arg0
	 */
	public SqlCursorTest(String arg0) {
		super(arg0);
	}

	public static void main(String[] args) {

		junit.swingui.TestRunner.run(SqlCursorTest.class);
	}

	/**
	 * @see TestCase#setUp()
	 */
	protected void setUp() throws Exception {
		LocalConnector connector=new LocalConnector(
			"oracle.jdbc.driver.OracleDriver",
			" jdbc:oracle:thin:@arthur:1521:merlin",
			"scott",
			"miauw");
		m_connection=connector.getConnection();
	}

	/**
	 * @see junit.framework.TestCase#tearDown()
	 */
	protected void tearDown() throws Exception {

	}

	/*
	 * Test for void Cursor(Connection, String, String)
	 */
	public void testCursorConnectionStringString() {
		Cursor cursor =
			new SqlCursor(
				m_connection,
				"TEST",
				"SELECT ENAME FROM EMP WHERE HIREDATE<'31-DEC-81'");
		assertEquals(
			"SELECT ENAME FROM EMP WHERE HIREDATE<'31-DEC-81'",
			cursor.toString());

	}

	public void test0Select() {
		Cursor cursor =
			new SqlCursor(
				m_connection,
				"TEST",
				"SELECT ENAME FROM EMP WHERE HIREDATE<'31-DEC-81'");
		try {
			List v = cursor.retrieveList();
			assertEquals(11, v.size());
		} catch (Exception ex) {
			ex.printStackTrace();
			assertTrue(ex.getMessage(), false);
		}
	}

	public void test1Select() {
		Cursor cursor =
			new SqlCursor(
				m_connection,
				"TEST",
				"Select SUM(SAL) As TOTAL From EMP WHERE HIREDATE<='31-DEC-82' ");
		try {
			List v = cursor.retrieveList();
			assertEquals(1, v.size());
			assertEquals(24925,((CursorData)v.get(0)).getIntValue("TOTAL"));
		} catch (Exception ex) {
			Logger.printException(ex);
			assertTrue(ex.getMessage(), false);
		}
	}

	public void test1Update() {
		Cursor cursor =
			new SqlCursor(
				m_connection,
				"TEST",
				"SELECT ENAME, SAL FROM EMP WHERE HIREDATE<'31-DEC-81'");
		try {
			List v = cursor.retrieveList();
			assertEquals(11, v.size());
			CursorData smith = (CursorData) v.get(0);
			int oldSal=smith.getIntValue("SAL");
			smith.setValue("SAL", new Integer(999));
			cursor.update(smith);
			v= cursor.retrieveList();
			CursorData aboutSmith=(CursorData) v.get(0);
			assertEquals(999,aboutSmith.getIntValue("SAL"));
			
			smith.setValue("SAL",new Integer(oldSal));
			aboutSmith.setValue("SAL",new Integer(oldSal));
			try {
				cursor.update(smith);
				assertTrue("Should return an Exception",false);
			} catch (DataException sqex){
				assertTrue(true);
			}
			cursor.update(aboutSmith);
		} catch (Exception ex) {
			assertTrue(ex.getMessage(), false);
		}
	}
	
	public void test0Group() {
		Cursor cursor =
			new SqlCursor(
				m_connection,
				"TEST",
				"SELECT DEPTNO, SUM(SAL) DEPT_SAL FROM EMP GROUP BY DEPTNO ORDER BY SUM(SAL) DESC");
		try {
			List v = cursor.retrieveList();
			assertEquals(3,v.size());
			CursorData highRoller=(CursorData) v.get(0);
			assertEquals(20,highRoller.getIntValue("DEPTNO"));
			assertEquals(10875,highRoller.getIntValue("DEPT_SAL"));
		} catch (Exception ex) {
			ex.printStackTrace();
			assertTrue(ex.getMessage(), false);

		}

	}

	public void testCursorDate() {
		try {
			SqlCursor cursor=new SqlCursor(m_connection, "TEST",
				"SELECT COUNT(EMPNO) AS LOYAL FROM EMP WHERE HIREDATE < ?");
			Calendar c=Calendar.getInstance();
			c.set(1981,11,31);
			cursor.addParameter(1,c.getTime());
			List v = cursor.retrieveList(null);
			CursorData cd=(CursorData) v.get(0);
			assertEquals(1,v.size());
			assertEquals(11,cd.getIntValue("LOYAL"));
		} catch (Exception ex) {
			ex.printStackTrace();
			assertTrue(ex.getClass() + " : " + ex.getMessage(),false);
		}
	}

	public void testDelete() {
		try {
			m_connection.setAutoCommit(false);
			SqlCursor cursor=new SqlCursor(m_connection, "TEST",
				"SELECT EMPNO,ENAME,SAL FROM EMP WHERE SAL < ?");
			cursor.addParameter(1,1000);
			List v = cursor.retrieveList();
			CursorData cd=(CursorData) v.get(0);
			assertEquals(7369,cd.getIntValue("EMPNO"));
			cursor.delete(cd);
			v = cursor.retrieveList();
			CursorData cd2=(CursorData) v.get(0);
			assertTrue(cd2.getIntValue("EMPNO")!=7369);
			assertTrue(!cd.equals(cd2));
		} catch (Exception ex) {
			ex.printStackTrace();
			assertTrue(ex.getClass() + " : " + ex.getMessage(),false);
		} finally {
			try {
				m_connection.rollback();
			} catch (SQLException sqex) {
				sqex.printStackTrace();
			}
		}
	}

	public void testGroup() {
			try {
				SqlCursor c=new SqlCursor(m_connection, "TEST",
					"SELECT DEPT.DEPTNO, DNAME AS DEPARTMENT, COUNT(EMPNO) EMPLOYEES FROM DEPT,EMP WHERE DEPT.DEPTNO=EMP.DEPTNO GROUP BY DEPT.DEPTNO, DNAME");
				List v=c.retrieveList();
				assertEquals(3,v.size());
				//DEPTNO=40 has no employees
				CursorData cd=(CursorData) v.get(0);
				assertEquals(3,cd.getColumnCount());
				assertEquals(10,cd.getIntValue("DEPTNO"));
				assertEquals("ACCOUNTING", cd.getValue("DEPARTMENT"));
				assertEquals(3,cd.getIntValue("EMPLOYEES"));
			} catch (Exception ex) {
				ex.printStackTrace();
				assertTrue(ex.getClass() + " : " + ex.getMessage(),false);
			}
	}

	/*
	 * Test for Vector select()
	 */
	public void testSelect() {
		try {
	    	SqlCursor c=new SqlCursor(m_connection, "TEST",
				"SELECT ENAME AS NAME, DNAME AS DEPARTMENT FROM EMP,DEPT WHERE EMP.DEPTNO=DEPT.DEPTNO AND EMP.EMPNO=?");
	    	c.addParameter(1,7788);
			List v=c.retrieveList();
			assertEquals(1,v.size());
			CursorData cd=(CursorData) v.get(0);
			assertEquals(2,cd.getColumnCount());
			assertEquals("NAME",cd.getColumn(0));
			assertEquals("DEPARTMENT",cd.getColumn(1));
			assertEquals("SCOTT",cd.getValue("NAME"));
			assertEquals("RESEARCH",cd.getValue("DEPARTMENT"));
		} catch (Exception ex) {
			assertTrue(ex.getClass() + " : " + ex.getMessage(),false);
		}
	}

	public void testUpdate() {
		try {
			m_connection.setAutoCommit(false);
			SqlCursor cursor=new SqlCursor(m_connection, "TEST",
				"SELECT EMPNO,ENAME,SAL FROM EMP WHERE SAL < ? ORDER BY EMPNO");
			cursor.addParameter(1,1000);
			List v = cursor.retrieveList();
			CursorData cd=(CursorData) v.get(0);
			assertEquals(7369,cd.getIntValue("EMPNO"));
			assertEquals(800,cd.getIntValue("SAL"));
			
			cd.setValue("SAL",new Integer(999));
			cursor.update(cd);
			v = cursor.retrieveList();
			CursorData cd2=(CursorData) v.get(0);
			assertEquals(7369,cd2.getIntValue("EMPNO"));
			assertEquals(999,cd2.getIntValue("SAL"));
		} catch (Exception ex) {
			ex.printStackTrace();
			assertTrue(ex.getClass() + " : " + ex.getMessage(),false);
		} finally {
			try {
				m_connection.rollback();
			} catch (SQLException sqex) {
				sqex.printStackTrace();
			}
		}
	}
}
