/* 	LobHelper.java
 *	Copyright (c) 2008, Brains2B.org
 *	
 *	Created by: Dennis Groenendijk
 *	Created on: Mar 21, 2008
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that  * the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions and 
 * the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
 * the following disclaimer in the documentation and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products derived from this software 
 * without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY 
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.brains2b.data.sql.lob;

import java.io.InputStream;
import java.sql.Blob;
import java.sql.Clob;

import org.brains2b.sql.meta.types.Stream;

/** Helper method to create Clob, Blob and Streams either on 
 * the basis of the objects supplied or for creating new Objects
 *
 * @author <A HREF="MAILTO:dennis@brains2b.nl">dennis@brains2b.nl</A>
 * @version 0.90 [Apr 9, 2008]
 */
public class LobHelper {

	/**
	 * create a new empty Clob
	 * @return Clob
	 */
	public static final Clob createClob() {
		return new ClobImpl();
	}
	
	/**
	 * create a wrapper around an existing Clob
	 * @param clob Clob
	 * @return Clob, actually the ClobImpl that act as a wrapper
	 */
	public static final Clob createClob(Clob clob) {
		if (clob==null) return new ClobImpl();
		return new ClobImpl(clob);
	}
	
	/**
	 * create a new empty Blob
	 * @return Blob
	 */
	public static final Blob createBlob() {
		return new BlobImpl();
	}
	
	/**
	 * create a wrapper around an existing Blob
	 * @param blob Blob
	 * @return Blob, actually the BlobImpl that act as a wrapper
	 */
	public static final Blob createBlob(Blob blob) {
		if (blob==null) return new BlobImpl();
		return new BlobImpl(blob);
	}
	
	/**
	 * create a Stream object for the InputStream from
	 * the database, that can then be used to change the content
	 * by using {@link Stream#getOutputStream()}
	 * @param is InputStream
	 * @param disconnected boolean
	 * @return {@link Stream}
	 */
	public static final Stream createStream(InputStream is, boolean disconnected) {
		if (disconnected) {
			DisconnectedStream dis = new DisconnectedStream();
			dis.cacheStream(is);
			return dis;
		} else {
			return new ConnectedStream(is);
		}
	}
	
	/**
	 * create a new empty Stream
	 * @return {@link Stream}
	 */
	public static final Stream createStream() {
		// just use disconnected stream, while input is empty anyway
		return new DisconnectedStream();
	}
}
