/* 	LobCache.java
 *	Copyright (c) 2008, Brains2B.org
 *	
 *	Created by: Dennis Groenendijk
 *	Created on: Mar 21, 2008
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that  * the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions and 
 * the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
 * the following disclaimer in the documentation and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products derived from this software 
 * without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY 
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.brains2b.data.sql.lob;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.sql.Blob;
import java.sql.SQLException;
import java.util.Arrays;

import org.brains2b.data.HashCodeHelper;

/**
 * LobCache is local way to store data for a Blob, Clob and ConnectedStream
 * <p>The data is stored in memory if the content is smaller than 4096 bytes,
 * otherwise the data is stored in a local file.</p>
 * @author <A HREF="MAILTO:dennis@brains2b.nl">dennis@brains2b.nl</A>
 * @version 0.90 [21-03-2008]
 */
public class LobCache implements Blob {
	
	final static int SIZE = 4096;

	private ByteBuffer m_buf = ByteBuffer.allocateDirect(SIZE);
	private File m_cache;
	private int m_hashCode = HashCodeHelper.INITIAL;
	private int m_size;
	
	public InputStream getBinaryStream() throws SQLException {
		try {
			if (m_cache!=null) return new FileInputStream(m_cache);
		} catch (FileNotFoundException fnex) {
			throw new SQLException("Cache file " + m_cache.getAbsolutePath() + " cannot be found");
		}
		
		return new InputStream() {
			public int read() throws IOException {
				if (available()>0) return m_buf.get();
				return -1;
			}
			
			public synchronized void reset() throws IOException {
				m_buf.reset();
			}
			
			public int available() throws IOException {
				return m_size-m_buf.position();
			}
		};
	}

	public byte[] getBytes(long pos, int length) throws SQLException {
		if (m_cache!=null) {
			try {
				return readCache(pos,length);
			} catch (IOException iex) {
				throw new SQLException(iex.getMessage());
			}
		} 
		byte[] out = new byte[length];
		m_buf.get(out,0,length);
		return out;
	}

	public long length() throws SQLException {
		if (m_cache!=null) return m_cache.length();
		return m_size;
	}

	public long position(Blob pattern, long start) throws SQLException {
		throw new SQLException("LobCache only supports position(byte[],start)");
	}

	public long position(byte[] pattern, long start) throws SQLException {
		if (m_cache!=null) {
			try {
				return findInCache(pattern,start);
			} catch (IOException iex) {
				throw new SQLException(iex.getMessage());
			}
			
		}
		byte[] b = new byte[m_size];
		m_buf.get(b);
		return find(b,m_size, pattern,start);
	}

	public OutputStream setBinaryStream(long pos) throws SQLException {
		truncate(pos);
		return new OutputStream() {
			public void write(int b) throws IOException {
				m_buf.put((byte)b);
				if (m_buf.remaining()<1) {
					synchronized (m_buf) {
						writeCache();
						m_buf.notify();
					}	
				}
			}
			
			public void flush() throws IOException {
				if (m_cache!=null) {
					writeCache();
				} else {
					m_size = m_buf.position();
					m_buf.position(0);
				}
			}
			
			public void close() throws IOException {
				flush();
			}
		};
	}

	public int setBytes(long pos, byte[] bytes, int offset, int len) throws SQLException {
		try {
			OutputStream os = setBinaryStream(pos);
			os.write(bytes, offset, len);
			os.flush();
		} catch (IOException iex) {
			throw new SQLException(iex.getMessage());
		}
		return len;
	}

	public int setBytes(long pos, byte[] bytes) throws SQLException {
		return setBytes(pos,bytes,0,bytes.length);
	}

	public void truncate(long len) throws SQLException {
		if (m_cache!=null && len>m_cache.length()+m_buf.capacity()) {
			throw new SQLException("Cannot write beyond current location of buffer");
		} else if (m_cache!=null && len<m_cache.length()) {
			try {
				RandomAccessFile rac =new RandomAccessFile(m_cache,"rw");
				rac.setLength(len);
			} catch (IOException iex) {
				throw new SQLException("Cannot set new starting point for OutputStream");
			}
		} else if (m_cache==null && len<m_buf.position()) {
			m_buf.position((int) len);
		}
		
	}
	
	private long findInCache(byte[] pattern, long start) throws IOException {
		/*if (m_cache.length()<start) {
			return find(m_buf.array(),pattern,start-m_cache.length());
		}*/
		long pos = 0;
		FileInputStream fis = new FileInputStream(m_cache);
		try {
			fis.skip(start);
			pos = start;
			byte[] b = new byte[SIZE];
			int len = 0; 
			while ((len = fis.read(b))!=-1) {
				long p = find(b,len, pattern,0);
				if (p!=-1L) {
					return pos+p;
				}
				pos+=SIZE;
			}
		} finally {
			fis.close();
		}
		return -1L;
	}
	
	private long find(byte[] b, int len, byte[] pattern, long start) {
		long pos =-1;
		int l =0;
		for (int i= (int) start; i<len;i++) {
			if (b[i]==pattern[l++]) {
				if (l==1) pos=i;
				if (l>=pattern.length) return pos;
			} else {
				l=0;
			}
		}
		return -1;
	}
	
	private byte[] readCache(long pos, int len) throws IOException {
		byte[] out = new byte[len];
		FileInputStream fis = new FileInputStream(m_cache);
		fis.skip(pos);
		fis.read(out);
		return out;
	}
	
	private void writeCache() throws IOException {
		if (m_cache==null) {
			m_cache = File.createTempFile("lob", ".tmp");
			m_cache.deleteOnExit();
		}
		
		RandomAccessFile fos = new RandomAccessFile(m_cache,"rw");
		fos.seek(fos.length());
		byte[] b = new byte[m_buf.position()];
		m_buf.position(0);
		m_buf.get(b);
		m_hashCode +=Arrays.hashCode(b);
		fos.write(b,0,b.length);
		fos.close();
		m_buf.clear();
	}
	
	protected void finalize() throws Throwable {
		if (m_cache!=null && m_cache.exists()) {
			m_cache.delete();
		}
		super.finalize();
	}
	
	public int hashCode() {
		if (m_cache!=null) return m_hashCode;
		return m_buf.hashCode();
	}
}
