package org.brains2b.data.sql.lob;

import java.io.InputStream;
import java.io.OutputStream;
import java.sql.SQLException;

import org.brains2b.data.HashCodeHelper;
import org.brains2b.sql.meta.types.Stream;

/**
 * Implementation of Stream for database streams that are not connected
 * to the stream in the database.
 * <p>This implementation exists while some database (f.e. Oracle) will only
 * allow access to the stream from the exact position within a ResultSet</p>
 * <p>This method is the simplest way to keep this stream by using LobCache as
 * store the current content of the stream. <i>This is not an ideal solution. It
 * would be better to keep a cursor inside this stream, but this would require
 * information on the unique identifiers of a record to retrieve the stream
 * when requested</i></p>
 *
 * @author <A HREF="MAILTO:dennis@brains2b.nl">dennis@brains2b.nl</A>
 * @version 0.90 [21-03-2008]
 */
public class DisconnectedStream implements Stream {

	private LobCache m_cache;
	
	public DisconnectedStream() {
		super();
		m_cache = new LobCache();
	}
	
	public DisconnectedStream(InputStream is) {
		super();
		m_cache = new LobCache();
		cacheStream(is);
	}
	
	public InputStream getInputStream() throws SQLException {
		return m_cache.getBinaryStream();		
	}

	public OutputStream getOutputStream() throws SQLException {
		return m_cache.setBinaryStream(0);
	}

	public boolean isAccessed() throws SQLException {
		return true;
	}
	
	void cacheStream(InputStream is) {
		try {
			byte[] b = new byte[LobCache.SIZE];
			int l=0;
			OutputStream os = m_cache.setBinaryStream(0);
			while ((l=is.read(b))!=-1) {
				os.write(b, 0, l);
			}
			os.close();
			is.close();
		} catch (Exception ex) {
		}
	}
	
	public int hashCode() {
		return m_cache!=null?m_cache.hashCode():HashCodeHelper.INITIAL;
	}
}
