/* 	ClobImpl.java
 *	Copyright (c) 2008, Brains2B.org
 *	
 *	Created by: Dennis Groenendijk
 *	Created on: Mar 21, 2008
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that  * the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions and 
 * the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
 * the following disclaimer in the documentation and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products derived from this software 
 * without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY 
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.brains2b.data.sql.lob;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.sql.Clob;
import java.sql.SQLException;

import org.brains2b.data.HashCodeHelper;

/**
 * Implementation of Clob to access Clob's independently of the underlying
 * database
 * <p>This ColbImpl can both be used to create new Clobs and be a wrapper
 * around a Clob provided by a database. The wrapper is needed because not
 * all Database implementations have different ways on which one can interact
 * with the Clob</p>
 * @author <A HREF="MAILTO:dennis@brains2b.nl">dennis@brains2b.nl</A>
 * @version 0.90 [21-03-2008]
 * @see LobCache
 */

public class ClobImpl implements Clob {

	private Clob m_clob;
	private LobCache m_internal;
	
	public ClobImpl() {
		super();
		m_internal = new LobCache();
	}
	
	public ClobImpl(Clob blob) {
		super();
		m_clob = blob;
	}
	
	public InputStream getAsciiStream() throws SQLException {
		if (m_clob!=null) {
			return m_clob.getAsciiStream();
		} 
		return m_internal.getBinaryStream();
		
	}
	
	public Reader getCharacterStream() throws SQLException {
		return new InputStreamReader(getAsciiStream());
	}
	
	public String getSubString(long pos, int length) throws SQLException {
		if (m_clob!=null) {
			return m_clob.getSubString(pos, length);
		}
		return new String(m_internal.getBytes(pos,length));
	}

	public long length() throws SQLException {
		if (m_clob!=null) {
			return m_clob.length();
		}
		return m_internal.length();
	}

	public long position(String pattern, long start) throws SQLException {
		if (m_clob!=null) {
			return m_clob.position(pattern,start);
		}
		return m_internal.position(pattern.getBytes(),start*2);
	}

	public long position(Clob pattern, long start) throws SQLException {
		if (m_clob!=null) {
			return m_clob.position(pattern,start);
		}
		return m_internal.position(pattern.getSubString(0, (int)pattern.length()).getBytes() ,start*2);
	}

	public Writer setCharacterStream(long pos) throws SQLException {
		return new OutputStreamWriter(setAsciiStream(pos));
	}
	
	public OutputStream setAsciiStream(long pos) throws SQLException {
		if (m_clob!=null) {
			try {
				return m_clob.setAsciiStream(pos);
			} catch (SQLException sqex) {
				return getInternalStream(pos);
			}
		}
		return m_internal.setBinaryStream(pos*2);
	}

	public int setString(long pos, String str) throws SQLException {
		if (m_clob!=null) {
			try {
				return m_clob.setString(pos,str);
			} catch (SQLException sqex) {
				return setInternalString(pos,str,0,(int) m_clob.length());
			}
		}
		return m_internal.setBytes(pos,str.getBytes(),0, str.length());
	}

	public int setString(long pos, String str, int offset, int len) throws SQLException {
		if (m_clob!=null) {
			try {
				return m_clob.setString(pos,str,offset,len);
			} catch (SQLException sqex) {
				return setInternalString(pos,str,offset,len);
			}
		}
		return m_internal.setBytes(pos,str.substring(offset,len).getBytes());
	}

	public void truncate(long len) throws SQLException {
		if (m_clob!=null) {
			if (len<0 || len>m_clob.length()) return;
			try {
				m_clob.truncate(len);
			} catch (SQLException sqex) {
				setInternalString(0,getSubString(0,(int) len),0,(int) m_clob.length());
			}
		}
		if (len<0 || len>m_internal.length()) return;
		m_internal.setBytes(0,getSubString(0, (int) len).getBytes(),0, (int) len);

	}

	public int hashCode() {
		if (m_clob!=null) {
			return HashCodeHelper.INITIAL;
		}
		return m_internal.hashCode();
	}
	
	private OutputStream getInternalStream(long pos) throws SQLException {
		makeLobCache();
		return m_internal.setBinaryStream(pos);
	}
	
	private int setInternalString(long pos, String s,int offset, int len) throws SQLException {
		makeLobCache();
		final String sb = s.substring(offset,len);
		return m_internal.setBytes(pos*2, sb.getBytes(), 0, sb.length()*2);
	}
	
	private synchronized void makeLobCache() throws SQLException {
		m_internal = new LobCache();
		byte[] b= new byte[LobCache.SIZE];
		int len = 0;
		OutputStream os = m_internal.setBinaryStream(0L);
		InputStream is = m_clob.getAsciiStream();
		try {
			while ( (len=is.read(b))!=-1) {
				os.write(b, 0, len);		
			}
			os.close();
			is.close();
			m_clob = null;
		} catch (IOException iex) {
			throw new SQLException(iex.getMessage());
		} 
	}
}
