/* 	BlobImpl.java
 *	Copyright (c) 2008, Brains2B.org
 *	
 *	Created by: Dennis Groenendijk
 *	Created on: Mar 21, 2008
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that  * the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions and 
 * the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
 * the following disclaimer in the documentation and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products derived from this software 
 * without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY 
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.brains2b.data.sql.lob;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.sql.Blob;
import java.sql.SQLException;

import org.brains2b.data.HashCodeHelper;

/**
 * Implementation of Blob to access Blob's independently of the underlying
 * database
 * <p>This ColbImpl can both be used to create new Blobs and be a wrapper
 * around a Clob provided by a database. The wrapper is needed because not
 * all Database implementations have different ways on which one can interact
 * with the Blob</p>
 * @author <A HREF="MAILTO:dennis@brains2b.nl">dennis@brains2b.nl</A>
 * @version 0.90 [21-03-2008]
 * @see LobCache
 */
public class BlobImpl implements Blob {

	private Blob m_blob;
	private LobCache m_internal;
	
	public BlobImpl() {
		super();
		m_internal = new LobCache();
	}
	
	public BlobImpl(Blob blob) {
		super();
		m_blob = blob;
	}
	
	public InputStream getBinaryStream() throws SQLException {
		if (m_blob!=null) {
			return m_blob.getBinaryStream();
		} 
		return m_internal.getBinaryStream();
		
	}

	public byte[] getBytes(long pos, int length) throws SQLException {
		if (m_blob!=null) {
			return m_blob.getBytes(pos, length);
		}
		return m_internal.getBytes(pos,length);
	}

	public long length() throws SQLException {
		if (m_blob!=null) {
			return m_blob.length();
		}
		return m_internal.length();
	}

	public long position(byte[] pattern, long start) throws SQLException {
		if (m_blob!=null) {
			return m_blob.position(pattern,start);
		}
		return m_internal.position(pattern,start);
	}

	public long position(Blob pattern, long start) throws SQLException {
		if (m_blob!=null) {
			return m_blob.position(pattern,start);
		}
		return m_internal.position(pattern.getBytes(0, (int)pattern.length()) ,start);
	}

	public OutputStream setBinaryStream(long pos) throws SQLException {
		if (m_blob!=null) {
			try {
				return m_blob.setBinaryStream(pos);
			} catch (SQLException sqex) {
				return getInternalStream(pos);
			}
		}
		return m_internal.setBinaryStream(pos);
	}

	public int setBytes(long pos, byte[] bytes) throws SQLException {
		if (m_blob!=null) {
			try {
				return m_blob.setBytes(pos,bytes);
			} catch (SQLException sqex) {
				return setInternalBytes(pos,bytes,0,(int) m_blob.length());
			}
		}
		return m_internal.setBytes(pos,bytes,0,bytes.length);
	}

	public int setBytes(long pos, byte[] bytes, int offset, int len)
			throws SQLException {
		if (m_blob!=null) {
			try {
				return m_blob.setBytes(pos,bytes,offset,len);
			} catch (SQLException sqex) {
				return setInternalBytes(pos,bytes,0,(int) m_blob.length());
			}
		}
		return m_internal.setBytes(pos,bytes,0,(int) m_blob.length());
	}

	public void truncate(long len) throws SQLException {
		if (m_blob!=null) {
			try {
				m_blob.truncate(len);
			} catch (SQLException sqex) {
				setInternalBytes(0,getBytes(0, (int) len),0,(int) m_blob.length());
			}
		}
		m_internal.setBytes(0,getBytes(0, (int) len),0,(int) m_blob.length());

	}

	public int hashCode() {
		if (m_blob!=null) {
			return HashCodeHelper.INITIAL;
		}
		return m_internal.hashCode();
	}
	
	private OutputStream getInternalStream(long pos) throws SQLException {
		makeLobCache();
		return m_internal.setBinaryStream(pos);
	}
	
	private int setInternalBytes(long pos, byte[] b,int offset, int len) throws SQLException {
		makeLobCache();
		return m_internal.setBytes(pos, b, offset, len);
	}
	
	private synchronized void makeLobCache() throws SQLException {
		m_internal = new LobCache();
		byte[] b= new byte[LobCache.SIZE];
		int len = 0;
		OutputStream os = m_blob.setBinaryStream(0L);
		InputStream is = m_blob.getBinaryStream();
		try {
			while ( (len=is.read(b))!=-1) {
				os.write(b, 0, len);		
			}
			os.close();
			is.close();
			m_blob = null;
		} catch (IOException iex) {
			throw new SQLException(iex.getMessage());
		} 
	}
}
