/* 	SqlCursor.java
 *	Copyright (c) 2003-2008, Brains2B.org
 *	
 *	Created by: Dennis Groenendijk
 *	Created on: 04-08-2003
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that  * the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions and 
 * the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
 * the following disclaimer in the documentation and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products derived from this software 
 * without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY 
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/* @#History
 * [When]		[Who]				[What]
 * 16-01-2008	dennis@brains2b.nl	Renamed OracleCondition into SqlCondition,Waldorf changes
 * 21-03-2008	dennis@brains2b.org	Support for reading LOB's and Streams			
 */
package org.brains2b.data.sql.cursor;

import java.math.BigDecimal;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Vector;

import org.brains2b.data.Condition;
import org.brains2b.data.DataException;
import org.brains2b.data.cursor.Cursor;
import org.brains2b.data.cursor.CursorData;
import org.brains2b.data.sql.SqlCondition;
import org.brains2b.data.sql.lob.LobHelper;
import org.brains2b.log.Logger;
import org.brains2b.sql.meta.SelectDescriptor;
import org.brains2b.sql.meta.TypesInfo;
import org.brains2b.sql.meta.types.Stream;
import org.brains2b.util.StringHelper;

/** Implementation of the Cursor object for Sql queries and add functionality for having queries with parameters.
 * <p>The sql statement can contain placeholder characters <code>?</code> signifying a
 * parameter which can be set at runtime by calling addParameter(index, value). The index 
 * corresponds with the count of the placeholder character in the SQL statement starting at <b>1</b>
 * 
 * @author <A HREF="MAILTO:dennis@brains2b.nl">dennis@brains2b.nl</A>
 * @version 0.25 [21-03-2008]
 */
public class SqlCursor extends Cursor {

	protected SelectDescriptor m_select;
	protected PreparedStatement m_prep;
	protected Connection m_con;
    
	private ArrayList m_parameters;
	private TypesInfo m_typeInfo;
	private boolean m_resultClosed;

	/**
	 * Constructor for SqlCursor
	 * @param name String, the name of the cursor
	 * @param sql String, the SQL-statement to use for this cursor
	 */
	public SqlCursor(String name, String sql) {
		super(name);
		m_select = new SelectDescriptor(sql);
		
	}

	/**
     * Constructor for SqlCursor.
     * <p>Usually the Connection does not have to be set at initialization, you can
     * use setConnection(Connection)
     * @param con java.sql.Connection, the JDBC connection to use for this Cursor
     * @param name String, the name of the cursor
	 * @param sql String, the SQL-statement to use for this cursor
     */
    public SqlCursor(Connection con, String name, String sql) {
        super(name);
        m_con = con;
        m_select = new SelectDescriptor(sql);
    }

    public CursorData retrieve(Condition con) throws DataException {
	    List l = retrieveList(con);
	    if (l==null) return null;
	    return (CursorData) l.get(0);
	}
	
    protected CursorData getObject(CursorData prototype, ResultSet rs) throws SQLException {
		CursorData cd= (CursorData) prototype.clone();
		cd.setRowNo(rs.getRow());
		for (int i=0;i<cd.getColumnCount();i++) {
			if (cd.getColumnType(i) == Blob.class) {
				cd.setValue(i, LobHelper.createBlob( (Blob) rs.getObject(i+1))); 	
			} else if (cd.getColumnType(i) == Clob.class) {
				cd.setValue(i, LobHelper.createClob( (Clob) rs.getObject(i+1)));
			} else if (cd.getColumnType(i) == Stream.class) {
				cd.setValue(i, LobHelper.createStream(rs.getBinaryStream(i+1), m_resultClosed));
			} else {
				cd.setValue(i,rs.getObject(i+1));
			}
		}
		cd.setChecksum(cd.hashCode());
		return cd;		
	}
    
	public CursorData getProtoType() throws DataException {
		if (m_prototype!=null) {
			return (CursorData) m_prototype.cloneAll();
		}
		try {
	        
	        StringBuffer sb=new StringBuffer();
	        sb.append(m_select.getColumns()).append(m_select.getFrom());
    		if (m_select.getWhere()!=null) sb.append(m_select.getWhere());
    		String s= getTypeInfo().getLimitRows(1);
    		if (sb.indexOf("WHERE")!=-1) {
    			s=s.replaceFirst("WHERE","AND");
    		}
    		sb.append(s);
    		if (m_select.getGroupBy()!=null) {
    			sb.append(" ").append(m_select.getGroupBy());
    		}
    		String sql = sb.toString();
    		if (getPlaceHolderCount()>0) {
				sql = sql.replaceAll("\\?","NULL");
	        }
	        m_prep=m_con.prepareStatement(sql);
			
			ResultSet rs=m_prep.executeQuery();
			return getProtoType(rs.getMetaData());
			
		} catch (SQLException sqex) {
			throw new DataException(DataException.RETRIEVE_FAILED,sqex);
		} finally {
			try {
				m_resultClosed = false;
				m_prep.close();
			} catch (SQLException sqex){
			}
		}
		
	}
	
	public List retrieveList() throws DataException {
		return retrieveList(null);
	}
	
	public List retrieveList(Condition condition) throws DataException {
	    Vector result=new Vector();
	    try {
	         if (condition==null) {
		    		m_prep=getPreparedStatement(m_con,m_select.toString());
	        } else if (condition!=null && condition instanceof SqlCondition){
	        		StringBuffer sb=new StringBuffer();
	            SqlCondition oc= (SqlCondition) condition;
		        	if (!StringHelper.isEmpty(m_select.getWhere())) sb.append(m_select.getWhere());
		        	if (m_select.getGroupBy()!=null) sb.append(m_select.getGroupBy());
		        	if (m_select.getHaving()!=null) sb.append(m_select.getHaving());
		        	if (m_select.getOrderBy()!=null) sb.append(m_select.getOrderBy());
		        	if (sb.length()>0) oc.setPreparedQuery(sb.toString());
		         ArrayList list = getParameters();
		        	for (int i = 0; i < list.size(); i++) {
		           		oc.addParameter(i+1,list.get(i));
		        	}
		        	m_prep=oc.getPreparedStatement(m_con,m_select.getColumns()+ m_select.getFrom());
	        }
			
			ResultSet rs=m_prep.executeQuery();
			CursorData prototype=getProtoType(rs.getMetaData());
			if (m_resultClosed) {
				rs.close();
				rs = m_prep.executeQuery();
			}
			while (rs.next()) {
				CursorData cdata=getObject(prototype,rs);
				
				result.add(cdata);
			}	
			rs.close();
			if (result.isEmpty()) {
				result=null;
			}
	    } catch (SQLException sqex) {
	    	throw new DataException(DataException.RETRIEVE_FAILED,sqex);
	    } finally {
	    	try {
	    		m_resultClosed=false;
    			m_prep.close();
    		} catch (SQLException sqex){}
	    }
		return result;	    
    }
    


    /* (non-Javadoc)
	 * @see org.brains2b.data.cursor.Cursor#update(org.brains2b.data.cursor.CursorData)
	 */
	public int update(CursorData cdata) throws DataException {
        try {
	        StringBuffer sb=new StringBuffer();
	    	sb.append("UPDATE ").append(m_select.getFrom().substring(m_select.getFrom().indexOf("FROM")+4).trim());
			sb.append(" SET ");    	
	    	for (int i=0;i<cdata.getColumnCount();i++) {
	    		if (i!=0) sb.append(", ");
	    		sb.append(cdata.getColumn(i)).append(" = ?");
	    	}
	    	
	    	sb.append(" WHERE ");
	    	
	    	List l=retrieveList(null);
	       
	    	CursorData oldData= null;
	    	boolean found=false;
	    	for (int i=0;i<l.size();i++) {
	    		oldData=(CursorData) l.get(i);
	    		if (oldData.hashCode()==cdata.getChecksum()) {
	    			found=true;			
	    			break;
	    		}
	    	}
	        cdata.hashCode();
	        oldData.hashCode();
	    	if (!found || oldData.getChecksum()!=cdata.getChecksum()) {
	    		throw new SQLException("cursor is no longer valid");
	    	}
	        
	    	int condcnt=0;
	    	for (int i=0;i<oldData.getColumnCount();i++) {
	            if (condcnt!=0 && oldData.getColumnType(i)!=Clob.class && oldData.getColumnType(i)!=Blob.class && oldData.getColumnType(i)!=Stream.class) sb.append(" AND ");
	    		if (oldData.getValue(i)!=null) {
	    		    if (oldData.getColumnType(i)!=Clob.class && oldData.getColumnType(i)!=Blob.class && oldData.getColumnType(i)!=Stream.class){
		    			sb.append(oldData.getColumn(i)).append(" = ?");
		    			condcnt++;
	    		    }
	    		} else {
	                sb.append(oldData.getColumn(i)).append(" IS NULL");      
	            }
	    	}
	    	
	    	Logger.println(sb.toString(),Logger.DEBUG);
	    	
	    	m_prep=m_con.prepareStatement(sb.toString());
			for (int i=0;i<cdata.getColumnCount();i++) {
			    if (cdata.getValue(i)!=null || ( cdata.getValue(i) instanceof Stream && ((Stream) cdata.getValue(i)).isAccessed())) {
			    	if (cdata.getColumnType(i) == Clob.class) {
			    		m_prep.setCharacterStream(i+1, ((Clob) cdata.getValue(i)).getCharacterStream(),0);	
			    	} else if (cdata.getColumnType(i) == Blob.class) {
			    		m_prep.setBinaryStream(i+1, ((Blob) cdata.getValue(i)).getBinaryStream(),0);
			    	} else if (cdata.getColumnType(i) == Stream.class) {
			    		m_prep.setBinaryStream(i+1, ((Stream) cdata.getValue(i)).getInputStream(), 0);
			    	} else {
	    				m_prep.setObject(i+1,cdata.getValue(i));
			    	}
				} else {
					m_prep.setNull(i+1,getTypeInfo().getDataType(cdata.getColumnType(i)));
				}
	    	}
			condcnt=0;
	    	for (int i=0;i<oldData.getColumnCount();i++) {
	    	    if (oldData.getValue(i)!=null && oldData.getColumnType(i)!=Clob.class && oldData.getColumnType(i)!=Blob.class && oldData.getColumnType(i)!=Stream.class) {
	    			m_prep.setObject((condcnt++)+1+cdata.getColumnCount(),oldData.getValue(i));
				} 
	    	}	
			
			int result=m_prep.executeUpdate();
			m_prep.close();
			cdata.setChecksum(cdata.hashCode());
			return result;
        } catch (SQLException sqex) {
        	throw new DataException(DataException.UPDATE_FAILED,sqex);
        }
    }



    /* (non-Javadoc)
	 * @see org.brains2b.data.cursor.Cursor#delete(org.brains2b.data.cursor.CursorData)
	 */
	public int delete(CursorData cdata) throws DataException {
    	try {
	    	StringBuffer sb=new StringBuffer();
	    	sb.append("DELETE ").append(m_select.getFrom());
	   		sb.append(" WHERE ");
	    	for (int i=0;i<cdata.getColumnCount();i++) {
	    		if (i!=0) sb.append(" AND ");
	            if (cdata.getValue(i)!=null) {
				 sb.append( cdata.getColumn(i)).append("= ?");
	            } else {
	                sb.append (cdata.getColumn(i)).append(" IS NULL");
	            }
	    	}
	    	Logger.println(sb.toString(),Logger.DEBUG);
	    	
	    	m_prep=m_con.prepareStatement(sb.toString());
			
	        int cnt=1;
			for (int i=0;i<cdata.getColumnCount();i++) {
				if (cdata.getValue(i)!=null) {
	    			m_prep.setObject(cnt++,cdata.getValue(i));
				} 
	    	}
				
			return m_prep.executeUpdate();
    	} catch (SQLException sqex) {
    		throw new DataException(DataException.DELETE_FAILED,sqex);
    	} finally {
    		try {
    			m_prep.close();
    		} catch (SQLException sqex){}
    	}
		
        
    }
    
	/**
	 * insert the cursor data in the table for this cursor
	 * @param cdata CursorData, data to insert
	 * @return int, the number of records inserted should always be one
	 * @throws Exception
	 * @since DataC 0.31
	 */
	public int insert(CursorData cdata) throws DataException {
        try {
			StringBuffer sb=new StringBuffer();
	    	sb.append("INSERT INTO ").append(m_select.getFrom().substring(m_select.getFrom().indexOf("FROM")+4).trim());
			sb.append(" ( ");
	        int cnt=0;
	    	for (int i=0;i<cdata.getColumnCount();i++) {
	    		if (cdata.getValue(i)!=null) {
	                 if (cnt!=0) sb.append(", ");
	    		     sb.append(cdata.getColumn(i));
	                 cnt++;
	            }
	    	}
	    	sb.append(" ) VALUES ( ");
	    	cnt=0;
	    	for (int i=0;i<cdata.getColumnCount();i++) {
	    		if (cdata.getValue(i)!=null || ( cdata.getValue(i) instanceof Stream && ((Stream) cdata.getValue(i)).isAccessed())) {
	                if (cnt!=0) sb.append(", ");
	                sb.append("?");
	                cnt++;
	            }
	    	}
	    	sb.append(" )");
	    	Logger.println(sb.toString(),Logger.DEBUG);
	    	
	        cnt=0;
	    	m_prep=m_con.prepareStatement(sb.toString());
			for (int i=0;i<cdata.getColumnCount();i++) {
	    		if (cdata.getValue(i)!=null) {
	    			if (cdata.getColumnType(i) == Clob.class) {
			    		m_prep.setCharacterStream(i+1, ((Clob) cdata.getValue(i)).getCharacterStream(),0);	
			    	} else if (cdata.getColumnType(i) == Blob.class) {
			    		m_prep.setBinaryStream(i+1, ((Blob) cdata.getValue(i)).getBinaryStream(),0);
			    	} else if (cdata.getColumnType(i) == Stream.class && ((Stream) cdata.getValue(i)).isAccessed()) {
			    		m_prep.setBinaryStream(i+1, ((Stream) cdata.getValue(i)).getInputStream(), 0);
			    	} else {
	    				m_prep.setObject(++cnt,cdata.getValue(i));
			    	}
				} 
	            /*else {
					m_prep.setNull(i+1,Types2Java.getDataType(cdata.getColumnType(i)));
				}*/
	    	}
	    	
			int result=m_prep.executeUpdate();
			m_prep.close();
			cdata.setChecksum(cdata.hashCode());
			return result;
        } catch (SQLException sqex) {
        	throw new DataException(DataException.INSERT_FAILED,sqex);
        } finally {
        	try {
    			m_prep.close();
    		} catch (SQLException sqex){}
        }
    }
	
    /**
     * adds a integer value for the given placeholder position in the
     * prepared statement.
     * @param idx int, the placeholder position
     * @param i int, The value to set in the prepared statement
     */
    public void addParameter(int idx, int i) {
        addParameter(idx, new Integer(i));
    }
    
    
     public void addParameter(int idx, long l) {
        addParameter(idx, new Long(l));
    }
     
      public void addParameter(int idx, double d) {
        addParameter(idx, new Double(d));
    }

    /**
     * adds a value for the given placeholder position in the
     * prepared statement.
     * @param idx int, the placeholder position
     * @param o Object, the value to set in the prepared statement.
     */
    public void addParameter(int idx, Object o) {
    	if (o instanceof Date) {
    		o = new Timestamp(((Date)o).getTime());
    	}
        if (m_parameters == null && getPlaceHolderCount() > 0) {
	       m_parameters = new ArrayList(getPlaceHolderCount());
	       
        }
        if (o == null)
            o = "NULL";
        if (m_parameters.size()<idx) {
        	m_parameters.ensureCapacity(getPlaceHolderCount());
        	m_parameters.add(idx -1 ,o);
        } else {
        	m_parameters.set(idx - 1,o);
        }

    }
	
	/**
	 * reset this Cursor by removing all previously set parameter values
	 * @see #addParameter(int, int)
	 * @see #addParameter(int, Object)
	 */
	public void reset() {
		m_parameters.clear();
	}


	/**
	 * get a List of the parameter values which were added. 
	 * <p>The index of the list -1 corresponds to the index of the parameter added  
	 * @return ArrayList, a List of Objects containing the value for the parameter
	 * @throws SQLException
	 * @see #addParameter(int, int)
	 * @see #addParameter(int, Object)
	 */
    private ArrayList getParameters() throws SQLException {
        ArrayList list = new ArrayList();
        if (m_parameters != null) {
            if (m_parameters.size() != getPlaceHolderCount()) {
                throw new SQLException("DataC: Not all variables bound");
            }
            list.addAll(m_parameters);
        }
        
        return list;
    }


    /**
	 * get the number of placeholder <code>?</code> which are in the query
	 * @return int, the number of <code>?</code> found in the query String
	 */
    private int getPlaceHolderCount() {
        if (m_select.toString() == null || m_select.toString().length() == 0)
            return -1;
        int cnt = 0;
        int idx = m_select.toString().indexOf('?');
        while (idx != -1) {
            cnt++;
            idx = m_select.toString().indexOf('?', idx + 1);
        }
        return cnt;
    }


    /**
     * returns a prepared statement build from
     * the prepared query set, the parameters added,
     * the conditions set and the order by supplied.
     * <p>If a prepared query has placeholders, the addParameters
     * must be called for every placeholder supplied.
     * @param con Connection, the connection for which to prepare the query
     * @param select String, the select containing the fields to retrieve and the table
     * <p>This should be a valid ANSI-SQL `SELECT` statement without any conditions.
     * @return PreparedStatement, the PreparedStatement created for execution
     * @throws SQLException, throws the Exception if generated by the prepareStatement
     * command in java.sql.Connection()
     * @see java.sql.Connection
     * @see java.sql.PreparedStatement
     */
    public PreparedStatement getPreparedStatement(
        Connection con,
        String select)
        throws SQLException {
        String sql = select;
        Logger.println(sql, Logger.DEBUG);
        PreparedStatement stmt = con.prepareStatement(sql);
        
        // statements for performance reasons.
        // Danger: A query could possibly retrieve more then
        // 5000 records.
        stmt.setMaxRows(5000);
        ArrayList list = getParameters();
        for (int i = 0; i < list.size(); i++) {
           stmt.setObject(i + 1, list.get(i));
        }
        return stmt;
    }
    
    /** set the <code>java.sql.Connection</code> to use for this Cursor
     * @param con {@link java.sql.Connection}
     */
    public void setConnection(Connection con) {
    	m_con=con;
    }
   
	/**
	 * get a prototype on the basis of the SQL-statement for this Cursor, which is defined
	 * in the ResultSetMetaData.
	 * @param rsMd java.sql.ResultSetMetaData, created on the basis of this cursor SQL-statement
	 * @return CursorData, with all the fields defined but no values.
	 * @throws SQLException, thrown if the database delivers an SQLException.
	 */
	protected CursorData getProtoType(ResultSetMetaData rsMd) throws SQLException {
		if (m_prototype==null) {
			m_prototype=new CursorData(getName());
			for (int i=0;i<rsMd.getColumnCount();i++) {
				
				int precision = 0;
				try {
					precision = rsMd.getPrecision(i+1);
				} catch (NumberFormatException nfex) {
				}
				
				m_prototype.setColumn(i,rsMd.getColumnLabel(i+1));
				m_prototype.setColumnType(i,getTypeInfo().getJavaType(
					rsMd.getColumnType(i+1),
					precision,
					rsMd.getScale(i+1)
				));
				if ((rsMd.getColumnType(i+1)==Types.LONGVARCHAR || rsMd.getColumnType(i+1) == Types.LONGVARBINARY) && "Oracle".equals(m_con.getMetaData().getDatabaseProductName())) {
					m_resultClosed = true;					
				}
				int scale =(rsMd.getScale(i+1)>0)?rsMd.getScale(i+1):0;
				int size = rsMd.getColumnDisplaySize(i+1);
				size = (precision==0)?(size<0?Integer.MAX_VALUE:size):precision;
				m_prototype.setValue(i,new BigDecimal("" + size+"."+scale));
			}
			
		}
		return m_prototype;
	}
    
    /**
	 * Overwritten to return the SQL statement that defines the Cursor
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return m_select.toString();
    }
    
    public Connection getConnection() {
        return m_con;
    }

    public SelectDescriptor getSelect() {
        return m_select;
    }
    
    private TypesInfo getTypeInfo() {
    	if (m_typeInfo==null) {
    		m_typeInfo = new TypesInfo(m_con);
    	}
    	return m_typeInfo;
    }
}
