/* 	SqlCondition.java
 *	Copyright (c) 2001, Dennis Groenendijk
 * 			      2002-2008 Brains2B.org
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that
 * the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions and 
 * the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
 * the following disclaimer in the documentation and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products derived from this software 
 * without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY 
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/* @#History
 * [When]		[Who]				[What]
 * 16-01-2008	dennis@brains2b.nl	Renamed OracleCondition into SqlCondition,Waldorf changes
 * 15-03-2008	dennis@brains2b.nl	building query was incorrectly done by truncating String			
 */
package org.brains2b.data.sql;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Iterator;

import org.brains2b.data.Condition;
import org.brains2b.log.Logger;

/** SqlCondition is a specific implementation of Condition
 *  for SQL databases.
 *
 * @author <A HREF="MAILTO:dennis@brains2b.nl">dennis@brains2b.nl</A>
 * @version 0.24.1 [15-03-2008]
 * @see Condition
 */
public class SqlCondition extends Condition {

    private String m_preparedQuery;
    private ArrayList m_parameters;

    /**
     * Default Constructor
     */
    public SqlCondition() {
        super();
    }

    /**
     * Constructor for SqlCondition
     */
    public SqlCondition(String orderBy) {
        super(orderBy);
    }

    /**
     * Constructor for SqlCondition
     */
    public SqlCondition(String field, Object value) {
        super(field, value);
    }

    /**
     * Constructor for SqlCondition
     */
    public SqlCondition(String field, Object value, String orderBy) {
        super(field, value, orderBy);
    }

    public Object getCondition() {
        String cond = makeCondition();
        String order = makeOrderBy();
        return (cond == null ? "" : cond) + (order == null ? "" : order);
    }

    /**
     * returns a prepared statement build from
     * the prepared query set, the parameters added,
     * the conditions set and the order by supplied.
     * <p>If a prepared query has placeholders, the addParameters
     * must be called for every placeholder supplied.
     * @param con Connection, the connection for which to prepare the query
     * @param select String, the select containing the fields to retrieve and the table
     * <p>This should be a valid ANSI-SQL `SELECT` statement without any conditions.
     * @return PreparedStatement, the PreparedStatement created for execution
     * @throws SQLException, throws the Exception if generated by the prepareStatement
     * command in java.sql.Connection()
     * @see java.sql.Connection
     * @see java.sql.PreparedStatement
     */
    public PreparedStatement getPreparedStatement(
        Connection con,
        String select)
        throws SQLException {
        String sql = buildSql(select);
        
        ArrayList list = getParameters();
        int cnt=0;
        for (Iterator it=list.iterator();it.hasNext();) {
        	   Object o=it.next();
           if ("NULL".equals(o)) {
        		sql=replaceParameter(sql, "?","NULL",cnt--);
        		it.remove();	   	
           } else if (o instanceof String) {
               sql=replaceParameter(sql, "?","\'"+ o +"\'",cnt--);
               it.remove();	   	
           }
           cnt++;
        }
        
        
        Logger.println(sql, Logger.DEBUG);
        PreparedStatement stmt = con.prepareStatement(sql);
        
        // statements for performance reasons.
        // Danger: A query could possibly retrieve more then
        // 5000 records.
        //stmt.setFetchSize(1000);
       
        stmt.setMaxRows(5000);
        //ArrayList list = getParameters();
        for (int i = 0; i < list.size(); i++) {
           	stmt.setObject(i + 1, list.get(i));
        }
        return stmt;
    }

	/**
	 * get a List of the parameter values which were added. 
	 * <p>The index of the list -1 corresponds to the index of the parameter added  
	 * @return ArrayList, a List of Objects containing the value for the parameter
	 * @throws SQLException
	 * @see #addParameter(int, int)
	 * @see #addParameter(int, Object)
	 */
    private ArrayList getParameters() throws SQLException {
        ArrayList list = new ArrayList();
        if (m_parameters != null) {
            if (m_parameters.size() != getPlaceHolderCount()) {
                throw new SQLException("DataC: Not all variables bound");
            }
            list.addAll(m_parameters);
        }
        if (m_cond != null) {
            list.addAll(m_cond.values());
        }
        return list;
    }

	/**
	 * build the SQL WHERE-statement from the normal condition defined for this condition.
	 * @param str, String, the possible prepared query which can contain parameters 
	 * @return String the new WHERE-statement for this condition
	 * @see #getPreparedStatement(Connection, String)
	 * @see #getPreparedQuery()
	 */
    private String buildSql(String str) {
        StringBuffer sbSql = new StringBuffer(str);
        boolean hasWhere = false;
        if (m_preparedQuery != null) {
            int wIdx=m_preparedQuery.toUpperCase().indexOf("WHERE ");
            int sIdx=m_preparedQuery.toUpperCase().indexOf("SELECT ");
            hasWhere =  wIdx!=-1 && (sIdx==-1 || sIdx>wIdx) ;
            if (hasWhere) {
            	sbSql.append(" WHERE (");
            	sbSql.append(m_preparedQuery.substring(m_preparedQuery.indexOf("WHERE ")+6));
            	sbSql.append(") ");
            } else {
            	sbSql.append(" WHERE (");
            	sbSql.append(m_preparedQuery).append(") ");
            	hasWhere=true;
            }
        }
        if (m_cond != null) {
            if (!hasWhere) {
                sbSql.append(" WHERE ");
            } else {
                sbSql.append(" AND ");
            }
            Iterator it = m_cond.keySet().iterator();
            while (it.hasNext()) {
                String key = (String) it.next();
                if (m_cond.get(key) instanceof String) {
	                String value = (String) m_cond.get(key);
    	            if (value.equals("NULL")) {
        	            sbSql.append(key).append(" IS NULL ");
            	        it.remove();
                	} else if (value.indexOf('%') != -1) {
                    	sbSql.append(key).append(" LIKE ?");
    	            } else {
                    	sbSql.append(key);
                    	sbSql.append(" = ? ");
                	}
                } else {
                    sbSql.append(key);
                    sbSql.append(" = ? ");
                }
                sbSql.append(" AND ");
            }
            sbSql.delete(sbSql.length() - 4, sbSql.length());
        }
        String orderBy = makeOrderBy();
        if (orderBy != null)
            sbSql.append(orderBy);

        return sbSql.toString();
    }

	/**
	 * makes an SQL WHERE-clause on the basis of the conditions which
	 * were added through <code>addCondition</code> but ignores possible
	 * parameters.
	 * @return String, the SQL WHERE-clause
	 * @see #addCondition(String, int)
	 * @see #addCondition(String, java.util.Date)
	 * @see #addCondition(String, Object)
	 */
    private String makeCondition() {
        StringBuffer sb = new StringBuffer(" WHERE ");
        if (m_cond != null) {
            Iterator it = m_cond.keySet().iterator();
            while (it.hasNext()) {
                String str = (String) it.next();
                sb.append(str);
                Object o = m_cond.get(str);

                if ("NULL".equals(o)) {
                    sb.append(" is ");
                } else if (
                    o instanceof String && ((String) o).indexOf('%') != -1) {
                    sb.append(" like ");
                } else {
                    sb.append(" = ");
                }
                if (o instanceof String && !"NULL".equals(o)) {
                    sb.append("\'" + o + "\'");
                } else {
                    sb.append(o);
                }
                sb.append(" AND ");
            }
            String out = sb.toString();
            out = out.substring(0, out.lastIndexOf("AND "));
            return out;
        } else {
            return null;
        }

    }
	
	/**
	 * makes an SQL clause for the order by statements added by <code>addOrderBy</code>
	 * @return String, an SQL statement containing the ORDER BY information
	 * @see #addOrderBy(String, boolean)
	 * @see #addOrderBy(String)
	 */
    private String makeOrderBy() {
        StringBuffer sb = new StringBuffer(" ORDER BY ");
        if (m_orderBy != null) {
            Iterator it = m_orderBy.iterator();
            int i = 0;
            while (it.hasNext()) {
                String str = (String) it.next();
                if (i++>0) sb.append(", ");
                sb.append(str);
                
            }
            return sb.toString();
            
        } else {
            return null;
        }
    }

    /**
     * set the SQL WHERE clause for this condition. Could be used in conjunction with
     * placeholder character <code>?</code> to later set parameters to.
     * <p>A prepared query is a query containing <b>only</b>
     * the query for a table, with or without the "WHERE" keyword.
     * @param preparedQuery The preparedQuery to set
     */
    public void setPreparedQuery(String preparedQuery) {
        m_preparedQuery = preparedQuery;

    }

    /**
     * get the SQL WHERE clause
     * @return String, the query String generated from
     * supplied condtions and prepared queries.
     */
    public String getPreparedQuery() {
        return m_preparedQuery;
    }

    /**
     * adds a value for the given placeholder position in the
     * prepared statement.
     * @param idx int, the placeholder position
     * @param o Object, the value to set in the prepared statement.
     */
    public void addParameter(int idx, Object o) {
        if (m_parameters == null && getPlaceHolderCount() > 0) {
		    m_parameters = new ArrayList(getPlaceHolderCount());
        }
        if (o == null) {
            o = "NULL";
        } else if (o instanceof java.util.Date) {
        	o=new Timestamp(((java.util.Date) o).getTime());
        }
        if (m_parameters.size()<idx) {
        	m_parameters.ensureCapacity(getPlaceHolderCount());
        	m_parameters.add(idx -1 ,o);
        } else {
        	m_parameters.set(idx - 1,o);
        }

    }

    /**
     * adds a integer value for the given placeholder position in the
     * prepared statement.
     * @param idx int, the placeholder position
     * @param i int, The value to set in the prepared statement
     */
    public void addParameter(int idx, int i) {
        addParameter(idx, new Integer(i));
    }

    /**
     * Implementation of addCondition which checks for <code>null</code> values
     * and makes them absolute so the condition will have a condition FIELD IS NULL
     * in the query.
     * <p>The normal <code>addCondition(String, Object)</code> does not do this by default and 
     * implementations can differ from data storage to data storage
     * <p>converts <code>java.util.Date</code> to <code>java.sql.Timestamp</code> which is 
     * appropriate for Oracle
     * @see #addCondition(String, Object)
     */
    public void addCondition(String field, Object value) {
        if (value == null) value = "NULL";
        if (value instanceof java.util.Date) {
        		value= new Timestamp( ((java.util.Date) value).getTime() );
        		((Timestamp) value).setNanos(0);
        } 
        
        super.addCondition(field, value);
    }
   
    /**
     * adds an Order By statement with the option to sort it
     * descending
     * @param field String, the field name as it occurs in the table
     * @param descending boolean, true if you want to sort descending.
     * @since DataC 0.23
     */
	public void addOrderBy(String field, boolean descending) {
        if(m_orderBy==null) {
            m_orderBy=new ArrayList();
        }
        m_orderBy.add(field + (descending?" DESC":""));
    }

	/**
	 * get the number of placeholder <code>?</code> which are in the prepared query
	 * @return int, the number of <code>?</code> found in the prepared query String
	 * @see #setPreparedQuery(String)
	 */
    private int getPlaceHolderCount() {
        if (m_preparedQuery == null || m_preparedQuery.length() == 0)
            return -1;
        int cnt = 0;
        int idx = m_preparedQuery.indexOf('?');
        while (idx != -1) {
            cnt++;
            idx = m_preparedQuery.indexOf('?', idx + 1);
        }
        return cnt;
    }
    
    
    /**
     * replace a param with a certain value in a String for a given position
     * @param sql, String to replace values in
     * @param param, the parameter to look for
     * @param value, the value to replace this value with
     * @param index, the n-occurence of <code>param</code> you want to replace, all
     * previous and later occurences are not replaced 
     * @return String, the original String with the param replaced with the value
     * @since DataC 0.23.1
     */
    private String replaceParameter(String sql, String param, String value, int index) {
    	StringBuffer sb=new StringBuffer();
    	int idx=0;
    	for (int i=0;i<index+1;i++) {
    		idx=sql.indexOf(param,idx+1);
    	}
    	sb.append(sql.substring(0,idx));
    	sb.append(value);
    	sb.append(sql.substring(idx+1));
    	return sb.toString();
    }
}
