/* 	CursorData.java
 * 
 * Created on: 5-4-2003
 * Created by: Dennis Groenendijk
 * 
 * Copyright (c) 2003-2008, Brains2B.org
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that
 * the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions and 
 * the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
 * the following disclaimer in the documentation and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products derived from this software 
 * without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY 
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/* @#History
 * [When]		[Who]				[What]
 * 16-01-2008	dennis@brains2b.nl	Waldorf changes			
 * TODO String names for columns are threated as case sensitive
 */
package org.brains2b.data.cursor;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

import org.brains2b.data.HashCodeHelper;
import org.brains2b.data.OptimisticLock;

/** Object for containing results from a Cursor
 *  
 * @see Cursor
 * @author <A HREF="MAILTO:dennis@brains2b.nl">dennis@brains2b.nl</A>
 * @version 0.22 [16-01-2008]
 */
public class CursorData implements OptimisticLock, Cloneable {
	
    private final int serialID=HashCodeHelper.INITIAL;
	private int m_checksum;
	private int m_rowNo;
	private String m_name;
	private List m_columns;
	private List m_values;
	private List m_columnTypes;

    /**
     * Constructor for CursorData.
     * @param name String, the name of the data source for information
     * purposes only
     */
    public CursorData(String name) {
        super();
        m_name=name;
        m_columns=new ArrayList();
        m_columnTypes=new ArrayList();
        m_values=new ArrayList();
        
    }

	/**
	 * get the number of fields in this cursor data object
	 * @return int, the number of columns
	 */
	public int getColumnCount() {
		return m_columns.size();
	}
    
    /**
     * set a column identified by name at the specified position
     * @param idx int, the index to set the column to
     * @param name String, the identifier for this column
     */
    public void setColumn(int idx, String name) {
    	setToList(m_columns, idx, name);
    }
    
    /**
     * get the column identifier for the column at the given index
     * @param idx int, the index of the column
     * @return String, the column identifier
     */
    public String getColumn(int idx) {
    	return (String) m_columns.get(idx);
    }
    
    /** 
     * set the Class identifying the type of content for this column
     * by index.
     * @param idx int, the index of the column
     * @param c Class, the class identifying the content for this column.
     * Use the Integer.TYPE, Boolean.TYPE, etc. for the primitives
     */
     public void setColumnType(int idx, Class c) {
    	setToList(m_columnTypes, idx, c);
    }
    
    /**
     * get the column type for the column identified by name
     * @param name String, the column identifier
     * @return Class, the Class-definition of the column, for primitives are returned
     * as Integer.TYPE, Double.TYPE, etc.
     * @since DataC 0.30
     */
    public Class getColumnType(String name) {
    	return getColumnType(getIndex(name));
    }
	
	/**
     * get the column type for the column given by the specified index
     * @param idx int, the index of the column
     * @return Class, the Class-definition of the column, for primitives are returned
     * as Integer.TYPE, Double.TYPE, etc.
     * @since DataC 0.30
     */
	public Class getColumnType(int idx) {
		if (idx<m_columnTypes.size()) {
			return (Class) m_columnTypes.get(idx);
		}
		return null;
    }
   
   	/**
   	 * set the content value for the specified index
   	 * @param idx int, the index of the column
   	 * @param value Object, the value for this specific column. The type of Object
   	 * should be consistent with the type of column you defined through <code>setColumnType(int,Class)</code>
   	 * @see #setColumnType(int, Class)
   	 */
   	public void setValue(int idx, Object value) {
        if (idx<0) return;
   		setToList(m_values, idx, value);
   	}

	/**
   	 * set the content value for the column identified by name
   	 * @param name String, the column identifier
   	 * @param value Object, the value for this specific column. The type of Object
   	 * should be consistent with the type of column you defined through <code>setColumnType(int,Class)</code>
   	 * @see #setColumnType(int, Class)
   	 */
	public void setValue(String name, Object value) {
		setValue(getIndex(name),value);	
	}   
	
	/** convinience method to set the column name and value at once at a given index
	 * @param idx int, the index of the column
	 * @param name String, the column identifier
	 * @param value Object, the value for this specific column. You can set the column type
	 * afterwards. The value and type should be consistent
	 */
	public void setColumnValue(int idx, String name, Object value) {
		setColumn( idx, name );
		setValue ( idx, value);
	} 

	/**
	 * get the value of the column at the given index as an Object
	 * @param idx int, the index of the column
	 * @return Object, the value of this column, will return <code>null</code> if no
	 * previous value was set 
	 */
	public Object getValue(int idx) {
	    if (idx>=0 && idx<m_values.size()) {
	        return m_values.get(idx);
	    } 
	    return null;
	}
	
	/**
	 * get the value of the column at the given index as a double
	 * @param idx int, the index of the column
	 * @return double, the double value for this column, 0 (zero) if the column
     * is not defined (yet)
	 * @throws NumberFormatException, If the value set to this specific column cannot be cast to an <code>int</code>
	 */
	public double getDoubleValue(int idx) {
	    if (getValue(idx)==null) return 0d;
		if (m_values.get(idx) instanceof BigDecimal) {
			return ((BigDecimal) m_values.get(idx)).doubleValue();
		} 
		throw new NumberFormatException("Value is not a number");
	}
	
	/**
     * get the double balue belonging to the column identified by name
     * @param name String, the name of the column
     * @return double, the value for this column, 0 (zero) if the column
     * is not defined (yet)
     * @throws NumberFormatException, If the value set to this specific column cannot be cast to an <code>int</code>
     */
	public double getDoubleValue(String name) {
		return getDoubleValue(getIndex(name));
	}
	
	/**
	 * get the value of the column at the given index as an integer.
	  @param idx int, the index of the column
	 * @return int, the int value for this column, 0 (zero) if the column
     * is not defined (yet)
	 * @throws NumberFormatException, If the value set to this specific column cannot be cast to an <code>int</code>
	 */
	public int getIntValue(int idx) {
		if (getValue(idx)==null) return 0;
		if (m_values.get(idx) instanceof BigDecimal) {
			return ((BigDecimal) m_values.get(idx)).intValue();
		} 
		throw new NumberFormatException("Value is not a number");
	}
	
	/**
	 * get the value of the column at the given index as a long.
	 * @param idx int, the index of the column
	 * @return long, the long value for this column, 0 (zero) if the column
     * is not defined (yet)
	 * @throws NumberFormatException, If the value set to this specific column cannot be cast to an <code>int</code>
	 * @since DataC 0.31
	 */
	public long getLongValue(int idx) {
	    if (getValue(idx)==null) return 0l;
		if (m_values.get(idx) instanceof BigDecimal) {
			return ((BigDecimal) m_values.get(idx)).longValue();
		} 
		throw new NumberFormatException("Value is not a number");
	}
	
	 /**
     * get the longValue belonging to the column identified by name
     * @param name String, the name of the column
     * @return long, the value for this column, 0 (zero) if the column
     * is not defined (yet)
     * @throws NumberFormatException, If the value set to this specific column cannot be cast to an <code>int</code>
     * @since DataC 0.31
     */
	public long getLongValue(String name) {
		return getLongValue(getIndex(name));
	}
	
	/**
	 * get the value of the column identified by name as an Object
	 * @param name String, the column identifier
	 * @return Object, the value for this column
	 */
	public Object getValue(String name) {
		return m_values.get(getIndex(name));
	}

	
    /**
     * get the intValue belonging to the column identified by name
     * @param name String, the name of the column
     * @return int, the value for this column, 0 (zero) if the column
     * is not defined (yet)
     * @throws NumberFormatException, If the value set to this specific column cannot be cast to an <code>int</code>
     */
	public int getIntValue(String name) {
		return getIntValue(getIndex(name));
	}

	/** get the index of the column identified by name
	 * @param name String, the column identifier
	 * @return int, the index of the column
	 */
	private int getIndex(String name) {
		for (int i=0;i<m_columns.size();i++) {
			if (m_columns.get(i).equals(name)) {
				return i;
			}
		}
		return -1;
	}
	
    public int getChecksum() {
        return m_checksum;
    }

    public void setChecksum(int value) {
    	m_checksum=value;
    }

	public int hashCode() {
        int result=HashCodeHelper.INITIAL;
        for (int i=0;i<m_columns.size();i++) {
	    	final Object o=getValue(i);
	    	if (o!=null) {
	    		result = HashCodeHelper.hash(result,o);
	    	}
	    }
        return result;
    }
    
    /**
     * clone will copy the column definitions, but will not clone the values set for this
     * CursorData.
     * <p>This is mostly used to get from a prototype to a real CursorData object
     * @see java.lang.Object#clone()
     */
    public Object clone() {
    	CursorData result=new CursorData(m_name);
    	result.m_columns=m_columns;
    	result.m_columnTypes=m_columnTypes;
    	return result;
    }
    
    /**
     * clone will copy the column definitions including the values
     */
    public Object cloneAll() {
    	CursorData result=new CursorData(m_name);
    	result.m_columns=m_columns;
    	result.m_columnTypes=m_columnTypes;
    	result.m_values=m_values;
    	return result;
    }


    /**
     * get the position of this CursorData object in a collection
     * <p>This value is set by the <code>select</code> procedure itself and does
     * not have to be consistent with <code>List.indexOf(Object)</code> and is used
     * to <code>update</code> a specific cursor object 
     * @return int, the row number of this CursorData object in a collection
     */
    public int getRowNo() {
        return m_rowNo;
    }

    /**
     * set the position of this CursorData object in a collection
     * <p>This value is set by the <code>select</code> procedure itself and does
     * not have to be consistent with <code>List.indexOf(Object)</code>. It is used
     * to <code>update</code> a specific cursor object 
     * @param rowNo int, the position to set
     */
    public void setRowNo(int rowNo) {
        m_rowNo = rowNo;
    }

    /**
     * get the name of the Cursor this CursorData object is a result for
     * @return String, the name of the Cursor
     */
    public String getName() {
        return m_name;
    }

    /**
     * set the name of the Cursor this CursorData object is a result for
     * @param name String, the name to set
     */
    public void setName(String name) {
        m_name = name;
    }

	/**
	 * determines if two objects are equal by comparing their hashCodes
     * @see java.lang.Object#equals(Object)
     */
    public boolean equals(Object obj) {
        return hashCode()==obj.hashCode();
    }

    public int serialID() {
        return serialID;
    }

    /**
     * set a value to a specific position in a List regardless
     * if this position already exists.
     * <p>If an object at this position exists the value is replaced
     * @param l List
     * @param idx int, the position to insert the Object into
     * @param o Object
     */
	private final void setToList(List l, int idx, Object o) {
		if (idx<l.size()) {
			l.set(idx,o);
		} else {
			for (int i=l.size();i<idx;i++) {
				l.add(i,null);
			}
			l.add(idx,o);
		}
	}

}
