/* 	Cursor.java
 * 
 * Created on: May 5, 2003
 * Created by: Dennis Groenendijk
 * 
 * Copyright (c) 2003-2008, Brains2B.org
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that
 * the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions and 
 * the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
 * the following disclaimer in the documentation and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products derived from this software 
 * without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY 
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
 /* @#History
 * [When]		[Who]				[What]
 * 16-01-2008	dennis@brains2b.nl	Waldorf changes			
 */
package org.brains2b.data.cursor;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import org.brains2b.data.Condition;
import org.brains2b.data.DataController;
import org.brains2b.data.DataException;
import org.brains2b.log.Logger;

/** Cursor is the Control class for a valid SQL select statement to be be used as
 * a data class for which we can manipulate the data. 
 * <p>This includes inserting, deleting or selecting data, a bit like an Oracle Cursor.
 * <p><b>Usage</b>: Create a new Cursor object with a valid SQL connection and a String
 * containing a valid SQL statement. <code>Cursor c=new Cursor({Valid Connection},"A distinquising name", "SELECT NAME, DEPARTMENT FROM EMP,DEP WHERE NAME.DEP_ID=DEPARTMENT.ID");</code>
 * After this a Select statement will return a CursorData object. See the javadoc for this object for more information.
 * To get a new CursorData object use <code>c.getNewCursorData()</code>
 * <p><b>Limitation</b>: Updating and inserting will only proceed if the underlying database allows this particular
 * operation for the particular query. Usually this only happens when all reference are explicit, all primary
 * keys are set and all non-null values are filled. Cursor will pass the Exception thrown by the database through
 * when there is a problem with mutating data.
 * 
 * @author <A HREF="MAILTO:dennis@brains2b.nl">dennis@brains2b.nl</A>
 * @version 0.23 [16-01-2008]
 * @see org.brains2b.data.cursor.CursorData
 */
public abstract class Cursor {

	private String m_name;
    protected CursorData m_prototype;
    
    /**
     * default constructor
     * @param name String, a name for this cursor
     */
    public Cursor(String name) {
    	super();
    	m_name = name;
    }
    
    /**
	 * Get an new CursorData object for this cursor, which has all the fields for this Cursor defined.
	 * @return CursorData, a new CursorData object which is basically a prototype retrieved from
	 * the database. Will return null if the underlying prototype cannot be returned, or if the database
	 * throws an Exception.
	 */
	public CursorData getNewCursorData() {
		if (m_prototype==null) {
			try {
				m_prototype= getProtoType();
			} catch (Exception ex) {
				Logger.printException(ex);
				return null;
			}
		} 
		return (CursorData) m_prototype.clone();
	}

	/**
	 * get a prototype for this cursor, which is basically a defintion
	 * of the columns and types, but without the values
	 * @return {@link CursorData}
	 * @throws DataException
	 */
	public abstract CursorData getProtoType() throws DataException;
		
	/**
	 * get a CursorData object filled from the current position in the ResultSet.
	 * @param prototype CursorData, object which functions as a prototype, will be
	 * cloned to function as return type. 
	 * @param rs ResultSet, a resultset with a valid position
	 * @return CursorData, an object with all fields and values set. 
	 * @throws SQLException, thrown if the resultset has no current position, or
	 * if the database delivers an SQLException
	 */
	protected CursorData getObject(CursorData prototype, ResultSet rs) throws SQLException {
		CursorData cd= (CursorData) prototype.clone();
		cd.setRowNo(rs.getRow());
		for (int i=0;i<cd.getColumnCount();i++) {
			cd.setValue(i,rs.getObject(i+1));
		}
		cd.setChecksum(cd.hashCode());
		return cd;		
	}

	/**
	 * get the name for this cursor
	 * @return String, the name of this cursor
	 */
    public String getName() {
        return m_name;
    }

    /**
	 * update an object of type o to the datastore
	 * @param cd CursorData, the object to update
	 * @return int, the number or records updated, should always return one
	 * @throws DataException, exception thrown from the specific data store controls, such
	 * as a JDBC driver are wrapped within this Exception
	 * @see DataController#update(Object)
	 */
	public abstract int update(CursorData cd) throws DataException;
	
	/**
	 * insert an object of type o to the datastore
	 * @param cd CursorData, the object to insert
	 * @return int, the number or records inserted, should always return one
	 * @throws DataException, exception thrown from the specific data store controls, such
	 * as a JDBC driver are wrapped within this Exception
	 * @see DataController#insert(Object)
	 */
	public abstract int insert(CursorData cd) throws DataException;
	
	/**
	 * delete an object of type o from the datastore
	 * @param cd CursorData, the object to delete
	 * @return int, the number or records deleted, should always return one
	 * @throws DataException, exception thrown from the specific data store controls, such
	 * as a JDBC driver are wrapped within this Exception
	 * @see DataController#delete(Object)
	 */
	public abstract int delete(CursorData cd) throws DataException;
	
	/**
	 * select an Object of the type of the <code>Object</code> o which match the supplied
	 * condition.
	 * @param condition Condition the condition which should be met by the retrieved objects
	 * @return An object which meets the condition. If more then one object meets the criteria
	 * implementations should only return the first one. Should return null if no object meets the
	 * condition.
	 * @throws DataException, exception thrown from the specific data store controls, such
	 * as a JDBC driver are wrapped within this Exception
	 * @see DataController#retrieve(Object, Condition)
	 * @see Condition
	 */
	public abstract CursorData retrieve(Condition condition) throws DataException;
	
	/**
	 * retrieve a list of Objects of the type of the <code>Object</code> o which match the supplied
	 * condition.
	 * @param condition Condition the condition which should be met by the retrieved objects
	 * @return A Vector of Objects which meet the condition. Should return null if no object meets the
	 * condition.
	 * @throws DataException, exception thrown from the specific data store controls, such
	 * as a JDBC driver are wrapped within this Exception
	 * @see DataController#retrieveList(Object, Condition)
	 * @see Condition
	 */
	public abstract List retrieveList(Condition condition) throws DataException;

	/**
	 * retrieve a list of Objects of the type of the <code>Object</code> 
	 * @return A list of CursorData which meet the defined cursor
	 * condition.
	 * @throws DataException, exception thrown from the specific data store controls, such
	 * as a JDBC driver are wrapped within this Exception
	 */
	public abstract List retrieveList() throws DataException;
}
