/* 	DataHelper.java
 *	Copyright (c) 2004, Brains2B.org
 *	
 *	Created by: Dennis Groenendijk
 *	Created on: Feb 17, 2004
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that  * the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions and 
 * the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
 * the following disclaimer in the documentation and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products derived from this software 
 * without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY 
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.brains2b.data;

import java.lang.reflect.Method;
import java.math.BigDecimal;

import org.brains2b.data.cursor.CursorData;
import org.brains2b.util.JavaHelper;

/**
 * DateHelper is a Library class which provides method(s) which provide support for 
 * working with data objects.
 * <p>All methods are static and should have no interdependancy between methods.
 * <p>See the javadoc for the individual methods for the uses of that function
 *
 * @author <A HREF="MAILTO:dennis@brains2b.nl">dennis@brains2b.nl</A>
 * @version 0.10 [17-02-2004]
 */
public final class DataHelper {

	/**
	 * copy the content of one data object to another. 
	 * <p>The objects do not have to be similar,
	 * but should be bean like in that they have getters and setters for the data they contain
	 * The data object data is copied by finding all the getters from the source and
	 * copy it to the setter of another object. This does mean datatypes should be equal.
	 * <p>There is no strictness is copying. All methods that follow these rules will be copied.
	 * If the source has more accessors than the target these members will be left empty. 
	 * If the target has less or dissimilar methods the data will be left as it was found.
	 * <p>getters should start with get and have no parameters, setters should start with set
	 * and take one parameter.
	 * @param source Object, where the data should be copied from, cannot be null
	 * @param target Object, where the data should be copied to, cannot be null
	 * @throws CopyException thrown if the data cannot be copied between objects usually this happens
	 * when accessors have the same name but use a different datatype.
	 */
	public static final void copyDataObject(Object source, Object target) throws CopyException {
		Method[] tms=target.getClass().getMethods();
		try {
			for (int i=0;i<tms.length;i++) {
				String name=tms[i].getName();
				if (name.startsWith("set") && !name.equals("setChecksum") && !name.equals("setRowid")) {
					String srcName="get" + name.substring(3);
					try {
						Method src=source.getClass().getMethod(srcName,null);
						Object o=src.invoke(source,null);
						tms[i].invoke(target,new Object[] {o});
					} catch (NoSuchMethodException nsmex) {
						//Just skip it
					}
				}
			}
		} catch (Exception ex) {
			throw new CopyException("copying " + source.getClass().getName() + " to " +target.getClass().getName()+ "failed");
		}
	}
	
	public static final void copyDataObject(CursorData source, Object target) throws CopyException {
		try {
			for (int i=0;i<source.getColumnCount();i++) {
				String name=source.getColumn(i);
				try {
					Method set=target.getClass().getMethod("set"+JavaHelper.objectName(name),new Class[] {source.getColumnType(i)});
					Object value= source.getValue(i);
					if (source.getColumnType(i)==Integer.TYPE) {
						value=new Integer(((BigDecimal) value).intValue());
					} else if (source.getColumnType(i)==Double.TYPE) {
						value=new Double(((BigDecimal) value).doubleValue());
					} else if (source.getColumnType(i)==Long.TYPE) {
						value=new Long(((BigDecimal) value).longValue());
					} 
					set.invoke(target,new Object[] {value});
					
				} catch (NoSuchMethodException nsmex) {
					//Just skip it
				}
			}
		} catch (Exception ex) {
			throw new CopyException("copying " + source.getClass().getName() + " to " +target.getClass().getName()+ "failed");
		}
	}

}
