/* 	DataController.java
 * 
 * Created on: Jun 19, 2002
 * Created by: Dennis Groenendijk
 * 
 * Copyright (c) 2002-2008, Brains2B.org
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that
 * the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions and 
 * the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
 * the following disclaimer in the documentation and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products derived from this software 
 * without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY 
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/* @#History
 * [When]		[Who]				[What]
 * 16-01-2008	dennis@brains2b.nl	Waldorf changes
 * TODO provide reference to working example.			
 */
package org.brains2b.data;

import java.util.List;

/** Interface for a general Controller to use for all different data objects.
 * <p>Implementation of this class should be able to find the DataControl implementation
 * for the type of Object which is provided as paramters through the methods
 * <p>Implementations would look something like:
 * <code><pre>
 * 		public int update(Object o) throws Exception {
 * 			return findDc(Object o).update();
 * 		}
 * 
 * 		public DataControl findDc(Object o) throws Exception {
 * 			//some method to find and instantiate the DataControl
 * 		}
 *</pre></code>
 * <p>This interface is defined in general terms because no assumptions should be made about the
 * type of datastore so the DataControl must be as general as possible. The most simple implementation would be using a JDBC connection, but you
 * could also use it for XML or flat file datastorage 
 * @author <A HREF="MAILTO:dennis@brains2b.nl">dennis@brains2b.nl</A>
 * @version 0.20 [16-01-2008]
 * @see org.brains2b.data.DataControl
 */
public interface DataController {
	
	/**
	 * update an object of type o to the datastore
	 * @param o Object, the object to update
	 * @return int, the number or records updated, should always return one
	 * @throws DataException, exception thrown if the specific data store controls throw an Exception,
	 * such as a JDBC driver.
	 * @see DataControl#update()
	 */
	public int update(Object o) throws DataException;
	
	/**
	 * insert an object of type o to the datastore
	 * @param o Object, the object to insert
	 * @return int, the number or records inserted, should always return one
	 * @throws DataException, exception thrown if the specific data store controls throw an Exception,
	 * such as a JDBC driver.
	 * @see DataControl#insert()
	 */
	public int insert(Object o) throws DataException;
	
	/**
	 * delete an object of type o from the datastore
	 * @param o Object, the object to delete
	 * @return int, the number or records deleted, should always return one
	 * @throws DataException, exception thrown if the specific data store controls throw an Exception,
	 * such as a JDBC driver.
	 * @see DataControl#update()
	 */
	public int delete(Object o) throws DataException;
	
	/**
	 * retrieve an Object of the type of the <code>Object</code> o which match the supplied
	 * condition.
	 * @param o Object the object which determines the type of objects you want to retrieve.
	 * An implementantion should not assume anything about the content of this Object and most 
	 * implementations will ignore the values.
	 * @param condition Condition the condition which should be met by the retrieved objects
	 * @return An object which meets the condition. If more then one object meets the criteria
	 * implementations should only return the first one. Should return null if no object meets the
	 * condition.
	 * @throws DataException, exception thrown if the specific data store controls throw an Exception,
	 * such as a JDBC driver.
	 * @see DataControl#retrieveList(Condition)
	 * @see Condition
	 */
	public Object retrieve(Object o, Condition condition) throws DataException;
	
	/**
	 * retrieve a list of Objects of the type of the <code>Object</code> o which match the supplied
	 * condition.
	 * @param o Object the object which determines the type of objects you want to retrieve.
	 * An implementantion should not assume anything about the content of this Object and most 
	 * implementations will ignore the values.
	 * @param condition Condition the condition which should be met by the retrieved objects
	 * @return A list of Objects which meet the condition. Should return null if no object meets the
	 * condition.
	 * @throws DataException, exception thrown if the specific data store controls throw an Exception,
	 * such as a JDBC driver.
	 * @see DataControl#retrieveList(Condition)
	 * @see Condition
	 */
	public List retrieveList(Object o, Condition condition) throws DataException;

}
	


