/* 	Condition.java
 *	Copyright (c) 2002-2008, Brains2B.org
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that
 * the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this list of conditions and 
 * the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
 * the following disclaimer in the documentation and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products derived from this software 
 * without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY 
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/* @#History
 * [When]		[Who]				[What]
 * 16-01-2008	dennis@brains2b.nl	Waldorf changes			
 */

package org.brains2b.data;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/** Abstract class defining a condition on a datastore.
 *  <p>The condition can be either an equals method for an identifier or
 *  an order by clause for your data.
 *  <p>Write an implementation for specific datastores by implementing
 *  the getCondition() to be executed as the query part for your default
 *  implementation.
 * 
 * @author <A HREF="MAILTO:dennis@brains2b.nl">dennis@brains2b.nl</A>
 * @version 0.23 [16-01-2008]
 */
public abstract class Condition  {
    
    protected List m_orderBy;
    protected HashMap m_cond;
    
	/** Default Constructor */
    public Condition() {
        super();
    }
    
    /**
     * Create a new Condition.
     * @param orderBy String the identifier you want to order the data by
     */
    public Condition(String orderBy) {
        super();
        addOrderBy(orderBy);
    }
    
     /** Create new Condition 
      * @param field String, the field identifier
      * @param value Object, value to look for
      * @see #addCondition(String,Object)
      */
    public Condition(String field, Object value) {
        super();
        addCondition(field,value);
    }
    
    /** Creates new Condition initialized with both an condition
     * and a sort order. 
     * <p>Kind of superflues, but it is here
     * @param field String, te identifier
     * @param value Object, the value to look for
     * @param orderBy String, identifier to sort by
     * */
    public Condition(String field, Object value, String orderBy) {
        super();
        addCondition(field,value);
        addOrderBy(orderBy);
    }

    /**
     * Adds a single statement to a condition of type <code>identifier equals value</code>
     * <p>Storage type assures you can only have one condition per identifier. In case
     * a field is set multiple times the last value will be used in the condition
     * @param field String, the identifier 
     * @param value Object, the value
     * @throws NullPointerException if value or field are empty
     */
    public void addCondition(String field, Object value) {
        if (m_cond==null && value!=null && field!=null) {
            m_cond=new HashMap();
        }
        m_cond.put(field, value);
    }
    
    /**
     * Adds a single statement to a condition of type <code>identifier equals value</code>
     * 
     * @param field String, the identifier 
     * @param value int, the value
     * @see #addCondition(String, Object)
     */
    public void addCondition(String field,int value) {
    	addCondition(field,new Integer(value));
    }
    
    /**
     * Adds a single statement to a condition of type <code>identifier equals value</code>
     * 
     * @param field String, the identifier 
     * @param value long, the value
     * @see #addCondition(String, Object)
     */
    public void addCondition(String field,long value) {
    	addCondition(field,new Long(value));
    }
    
    
    /**
     * removes the statement that was set for this identifier
     * @param field
     */
    public void removeCondition(String field) {
        if (m_cond==null) return;
        m_cond.remove(field);
        if (m_cond.isEmpty()) {
    		m_cond=null;
        }
    }
    
    
    /**
     * Adds a sort order to the condition.
     * <p>The order and sequence of the data will depend on the implementation.
     * Fields are stored in sequence they where added an multiple identifiers can be 
     * specified
     * @param field String, the column identifier
     */
    public void addOrderBy(String field) {
        if(m_orderBy==null) {
            m_orderBy=new ArrayList();
        }
        m_orderBy.add(field);
    }
    
    
    
    /**
     * removes previously set sort order. 
     * <p>No individual identifiers can be removed. Only all
     * values. You can then define a new set of order conditions.
     */
    public void clearOrderBy() {
    	m_orderBy=null;
    }
    
    
    /**
     * get list of order by statements
     * <p><i>Not happy with this function so expect it to change 
     * in future releases</i>
     * @return Vector, Strings containing the order by fields in
     * the sequence they where added, null if no order by clause was defined
     */
    public List getOrderBy() {
        return m_orderBy;
    }
    
    /**
     * get the statements added by the <code>addCondition</code> in a hashtable
     * <p><i>Not happy with this function so expect it to change 
     * in future releases</i> 
     * @return Hashtable, the fields and values as key, value pairs. Null if no
     * conditions where specified
     */
    public Map getConditions() {
        return m_cond;
    }
   
    /**
     * abstract function left to specific datasource implementation
     * to actually use condition and order by in a way the implementation
     * does understand
     * @return Object, most general implementation.
     */
    public abstract Object getCondition();
    
    /**
     * @see java.lang.Object#equals(Object)
     * @see java.lang.Object#hashCode()
     */
    public boolean equals(Object o) {
    	if (!(o instanceof Condition)) return false;
    	Condition c = (Condition) o;

    	if (m_cond==null && c.getConditions()!=null) {
    		return false;
    	} else if (m_cond!=null && c.getConditions()==null) {
    		return false;
    	} else if (m_cond!=null && c.getConditions()!=null && m_cond.hashCode()!=c.getConditions().hashCode()) {
    		return false;
    	} else if (m_orderBy==null && c.getOrderBy()!=null) {
    		return false;
    	} else if (m_orderBy!=null && !m_orderBy.equals(c.getOrderBy())) {
    		return false;
    	}
    	return true;
    }

}
